const express = require('express');
const dotenv = require('dotenv');
const cors = require('cors');
const jwt = require("jsonwebtoken");
const cookieParser = require("cookie-parser");
const swaggerUi = require('swagger-ui-express');
const swaggerJsdoc = require('swagger-jsdoc');
const connectDB = require('./config/db');

const authRoutes = require('./routes/authRoutes');
const workspaceRoutes = require('./routes/workspaceRoutes');
const sectionRoutes = require('./routes/sectionRoutes');
const taskRoutes = require('./routes/taskRoutes');
const subTaskRoutes = require('./routes/subTaskRoutes');
const adminRoutes = require('./routes/adminRoutes');
const groupRoutes = require('./routes/groupRoutes');
const calendarRoutes = require('./routes/calendarRoutes');
const titiRoutes = require('./routes/titiRoutes');
const dailySummaryRoutes = require('./routes/dailySummaryRoutes');
const morningMessageRoutes = require('./routes/morningMessageRoutes');
const premiumRoutes = require('./routes/premiumRoutes');
const discountRoutes = require('./routes/discountRoutes');
const couponRoutes = require('./routes/couponRoutes');
const courseRoutes = require('./routes/courseRoutes');
const assignmentRoutes = require('./routes/assignmentRoutes');

const helmet = require('helmet');
const { verifyTransporter } = require('./utils/emailService');

dotenv.config();
connectDB();

if (process.env.NODE_ENV !== "production") {
    verifyTransporter().catch(err => {
        console.warn("Email service verification skipped:", err.message);
    });
}

const app = express();

app.set('trust proxy', 1);

// ✅ Helmet relaxed (Namecheap health check needs basic headers)
app.use(helmet({ crossOriginResourcePolicy: false }));

app.use(cookieParser());

// ✅ Simple CORS mode for initial deployment (later restrict again)
app.use(cors({
    origin: true,
    credentials: true
}));

app.use(express.json());

// ✅ Namecheap health check routes
app.get("/", (req, res) => {
    res.setHeader("Content-Type", "text/html");
    res.send("<h2>TaskTree API Running ✅</h2>");
});

app.get("/status", (req, res) => {
    res.setHeader("Content-Type", "text/html");
    res.send("OK");
});

// Swagger config
const swaggerOptions = {
    definition: {
        openapi: "3.0.0",
        info: {
            title: "Notes/Tasks API",
            version: "1.0.0",
            description: "API for managing tasks, sections, and users",
        },
    },
    apis: ['./routes/*.js'],
};

const swaggerSpec = swaggerJsdoc(swaggerOptions);
app.use('/api-docs', swaggerUi.serve, swaggerUi.setup(swaggerSpec));

// ✅ API Routes
app.use('/api/auth', authRoutes);
app.use('/api/sections', sectionRoutes);
app.use('/api/tasks', taskRoutes);
app.use('/api/subtasks', subTaskRoutes);
app.use('/api/workspaces', workspaceRoutes);
app.use('/api/admin', adminRoutes);
app.use('/api/groups', groupRoutes);
app.use('/api/calendar', calendarRoutes);
app.use('/api/titi', titiRoutes);
app.use('/api/daily-summary', dailySummaryRoutes);
app.use('/api/morning-message', morningMessageRoutes);
app.use('/api/premium', premiumRoutes);
app.use('/api/discounts', discountRoutes);
app.use('/api/coupons', couponRoutes);
app.use('/api/courses', courseRoutes);
app.use('/api/assignments', assignmentRoutes);

// Daily automation scheduler (WhatsApp summaries at 23:00)
try {
    require('./dailyAutomation');
    console.log('✅ Daily automation module loaded.');
} catch (e) {
    console.warn('⚠️ Daily automation module not loaded:', e.message);
}

// Titi encouragement automation (3–4 random triggers per day)
try {
    require('./titiEncouragementAutomation');
    console.log('✅ Titi encouragement automation module loaded.');
} catch (e) {
    console.warn('⚠️ Titi encouragement automation not loaded:', e.message);
}

// Premium expiry reminders (daily at 9:00 AM)
try {
    require('./premiumExpiryReminder');
    console.log('✅ Premium expiry reminder module loaded.');
} catch (e) {
    console.warn('⚠️ Premium expiry reminder not loaded:', e.message);
}

// Global error handler
app.use((err, req, res, next) => {
    console.error(err.stack);
    res.status(500).send('Something broke!');
});

// ✅ MUST USE process.env.PORT for Namecheap
const PORT = process.env.PORT || 5000;
app.listen(PORT, () => console.log(`✅ Server running on PORT: ${PORT}`));
