const express = require('express');
const router = express.Router();
const groupController = require('../controllers/groupController');
const { protect } = require('../middleware/authMiddleware');
const { requirePremium } = require('../middleware/premiumMiddleware');

// Create group
router.post('/', protect, groupController.createGroup);
// List groups for user
router.get('/', protect, groupController.getGroups);
// Get group by ID (full details)
router.get('/:groupId', protect, groupController.getGroupDetails);
// Check if user emails exist
router.post('/check-emails', protect, groupController.checkUserEmails);
// Delete group and all related data
router.delete('/:groupId', protect, groupController.deleteGroup);
// Add a task to a group
router.post('/:groupId/tasks', protect, groupController.createGroupTask);
// Get all tasks for a group
router.get('/:groupId/tasks', protect, groupController.getGroupTasks);
// Update task status (completed/not completed)
router.patch('/:groupId/tasks/:taskId/status', protect, groupController.updateTaskStatus);
// Add a subtask to a group task
router.post('/:groupId/tasks/:taskId/subtasks', protect, groupController.addGroupSubtask);
// Update a subtask
router.put('/:groupId/tasks/:taskId/subtasks/:subtaskId', protect, groupController.updateGroupSubtask);
// Delete a subtask
router.delete('/:groupId/tasks/:taskId/subtasks/:subtaskId', protect, groupController.deleteGroupSubtask);
// Mark group task as done/undone
router.patch('/:groupId/tasks/:taskId/done', protect, groupController.markGroupTaskDone);
// Mark group subtask as done/undone
router.patch('/:groupId/tasks/:taskId/subtasks/:subtaskId/done', protect, groupController.markGroupSubtaskDone);
// Update a group task
router.put('/:groupId/tasks/:taskId', protect, groupController.updateGroupTask);
// Delete a group task
router.delete('/:groupId/tasks/:taskId', protect, groupController.deleteGroupTask);
// Update group details
router.put('/:groupId', protect, groupController.updateGroup);
// Add a member
router.post('/:groupId/members', protect, groupController.addGroupMember);
// Remove a member from a group
router.delete('/:groupId/members/:memberId', protect, groupController.removeGroupMember);
// Get group activity logs
router.get('/:groupId/activities', protect, groupController.getGroupActivityLogs);
// Get enhanced group analytics
router.get('/:groupId/analytics', protect, groupController.getGroupAnalytics);
// Get advanced analytics for a group (Premium only)
router.get('/:groupId/analytics/advanced', protect, requirePremium, groupController.getAdvancedAnalytics);
// Get all sections for a group
router.get('/:groupId/sections', protect, groupController.getGroupSections);
// Create a new section for a group
router.post('/:groupId/sections', protect, groupController.createGroupSection);
// Update a group section
router.put('/:groupId/sections/:sectionId', protect, groupController.updateGroupSection);
// Delete a group section
router.delete('/:groupId/sections/:sectionId', protect, groupController.deleteGroupSection);

module.exports = router;
