const express = require('express');
const router = express.Router();
const { protect } = require('../middleware/authMiddleware');
const adminController = require('../controllers/adminController');
const feedbackController = require('../controllers/feedbackController');

// Admin middleware to check if user is admin
const isAdmin = async (req, res, next) => {
    try {
        if (req.user.role !== 'admin') {
            return res.status(403).json({ message: 'Access denied. Admin only.' });
        }
        next();
    } catch (error) {
        res.status(500).json({ message: 'Error checking admin status', error: error.message });
    }
};

// Feedback submission is public (no auth)
router.post('/feedback', feedbackController.submitFeedback);

// Apply protection and admin check to all routes below
router.use(protect);
router.use(isAdmin);

// Dashboard routes
router.get('/dashboard/stats', adminController.getDashboardStats);
router.get('/dashboard/advanced-analytics', adminController.getAdvancedAnalytics);
// Automation reports
router.get('/automation/reports', adminController.getAutomationReports);
router.get('/automation/reports/:id', adminController.getAutomationReportDetails);
router.get('/users', adminController.getUsers);
router.put('/users/:userId/status', adminController.updateUserStatus);
router.post('/users/force-logout', adminController.forceLogout);
router.post('/users/reset-password', adminController.resetPassword);
router.post('/users/change-role', adminController.changeUserRole);
router.get('/activity-logs', adminController.getActivityLogs);
// Broadcast message to all users
router.post('/broadcast/message', adminController.broadcastMessage);
// Broadcast job with progress polling
router.post('/broadcast/start', adminController.startBroadcastJob);
router.get('/broadcast/progress/:jobId', adminController.getBroadcastProgress);
router.post('/broadcast/cancel', adminController.cancelBroadcastJob);
// Retry failed recipients and view detailed results
router.post('/broadcast/retry', adminController.retryBroadcastJob);
router.post('/broadcast/retry-specific', adminController.retryBroadcastSpecific);
router.get('/broadcast/results/:jobId', adminController.getBroadcastResults);

// Feedback viewing is admin-only
router.get('/feedbacks', feedbackController.getAllFeedbacks);
router.delete('/feedbacks/:id', feedbackController.deleteFeedback);

// Direct Premium Management (without activation requests)
router.post('/premium/assign', adminController.assignPremium);
router.post('/premium/remove', adminController.removePremium);
router.post('/premium/extend', adminController.extendPremium);

module.exports = router;