const mongoose = require('mongoose');

const activityLogSchema = new mongoose.Schema({
    userId: {
        type: mongoose.Schema.Types.ObjectId,
        ref: 'User',
        required: true,
        index: true
    },
    action: {
        type: String,
        required: true,
        enum: [
            'task_created',
            'task_completed',
            'task_assigned',
            'task_shared',
            'group_task_assigned',
            'group_task_created',
            'group_task_completed',
            'group_task_updated',
            'group_task_deleted',
            'group_member_added',
            'group_member_removed',
            'group_updated',
            'group_created',
            'workspace_created',
            'section_created'
        ]
    },
    taskTitle: {
        type: String,
        required: false
    },
    taskDescription: {
        type: String,
        required: false
    },
    taskId: {
        type: mongoose.Schema.Types.ObjectId,
        ref: 'Task',
        required: false
    },
    groupId: {
        type: mongoose.Schema.Types.ObjectId,
        ref: 'Group',
        required: false
    },
    groupName: {
        type: String,
        required: false
    },
    workspaceId: {
        type: mongoose.Schema.Types.ObjectId,
        ref: 'Workspace',
        required: false
    },
    date: {
        type: Date,
        default: Date.now,
        index: true
    },
    metadata: {
        type: mongoose.Schema.Types.Mixed,
        default: {}
    }
}, { timestamps: true });

// Index for efficient daily queries
activityLogSchema.index({ userId: 1, date: 1 });

// Static method to log activity
activityLogSchema.statics.logActivity = async function(activityData) {
    try {
        const activity = new this(activityData);
        await activity.save();
        return activity;
    } catch (error) {
        console.error('Error logging activity:', error);
        throw error;
    }
};

// Static method to get today's activities for a user
activityLogSchema.statics.getTodayActivities = async function(userId) {
    try {
        const startOfDay = new Date();
        startOfDay.setHours(0, 0, 0, 0);
        
        const endOfDay = new Date();
        endOfDay.setHours(23, 59, 59, 999);
        
        return await this.find({
            userId,
            date: {
                $gte: startOfDay,
                $lte: endOfDay
            }
        }).sort({ date: -1 });
    } catch (error) {
        console.error('Error getting today activities:', error);
        throw error;
    }
};

module.exports = mongoose.model('ActivityLog', activityLogSchema);