const User = require('../models/User');

/**
 * Middleware to check if user has active premium subscription
 */
const requirePremium = async (req, res, next) => {
    try {
        if (!req.user) {
            return res.status(401).json({
                success: false,
                message: 'Authentication required'
            });
        }

        const user = await User.findById(req.user._id);
        
        if (!user) {
            return res.status(404).json({
                success: false,
                message: 'User not found'
            });
        }

        // Check if premium is active and not expired
        const now = new Date();
        const isPremiumActive = user.premiumActive && 
                               user.premiumExpiry && 
                               new Date(user.premiumExpiry) > now;

        // Check if trial is active
        const isTrialActive = user.trialUsed && 
                             user.trialEndDate && 
                             new Date(user.trialEndDate) > now;

        // Allow access if premium OR trial is active
        if (!isPremiumActive && !isTrialActive) {
            // Auto-downgrade if expired
            if (user.premiumActive && user.premiumExpiry && new Date(user.premiumExpiry) <= now) {
                user.premiumActive = false;
                user.planType = 'free';
                await user.save();
            }

            return res.status(403).json({
                success: false,
                message: 'Premium subscription or trial required',
                code: 'PREMIUM_REQUIRED',
                premiumActive: false,
                premiumExpiry: user.premiumExpiry,
                trialActive: false
            });
        }

        // Add premium/trial info to request
        req.userPremium = {
            active: isPremiumActive || isTrialActive,
            expiry: user.premiumExpiry,
            planType: user.planType,
            isTrial: isTrialActive,
            trialEndDate: user.trialEndDate
        };

        next();
    } catch (error) {
        console.error('Premium middleware error:', error);
        res.status(500).json({
            success: false,
            message: 'Error checking premium status',
            error: error.message
        });
    }
};

/**
 * Middleware to check premium status without blocking (for feature flags)
 */
const checkPremium = async (req, res, next) => {
    try {
        if (!req.user) {
            req.isPremium = false;
            return next();
        }

        const user = await User.findById(req.user._id);
        
        if (!user) {
            req.isPremium = false;
            return next();
        }

        const now = new Date();
        const isPremiumActive = user.premiumActive && 
                               user.premiumExpiry && 
                               new Date(user.premiumExpiry) > now;

        // Check if trial is active
        const isTrialActive = user.trialUsed && 
                             user.trialEndDate && 
                             new Date(user.trialEndDate) > now;

        // Auto-downgrade if expired
        if (user.premiumActive && user.premiumExpiry && new Date(user.premiumExpiry) <= now) {
            user.premiumActive = false;
            user.planType = 'free';
            await user.save();
        }

        // User has premium access if premium OR trial is active
        req.isPremium = isPremiumActive || isTrialActive;
        req.userPremium = {
            active: isPremiumActive || isTrialActive,
            expiry: user.premiumExpiry,
            planType: user.planType,
            isTrial: isTrialActive,
            trialEndDate: user.trialEndDate
        };

        next();
    } catch (error) {
        console.error('Premium check middleware error:', error);
        req.isPremium = false;
        next();
    }
};

module.exports = { requirePremium, checkPremium };

