const Discount = require('../models/Discount');

/**
 * Get all discounts (admin only)
 */
const getAllDiscounts = async (req, res) => {
    try {
        const discounts = await Discount.find().sort({ createdAt: -1 });
        res.status(200).json({
            success: true,
            discounts
        });
    } catch (error) {
        console.error('Error fetching discounts:', error);
        res.status(500).json({
            success: false,
            message: 'Error fetching discounts',
            error: error.message
        });
    }
};

/**
 * Get active discounts
 */
const getActiveDiscounts = async (req, res) => {
    try {
        const discounts = await Discount.find({
            active: true,
            $or: [
                { endDate: { $exists: false } },
                { endDate: { $gte: new Date() } }
            ]
        }).sort({ createdAt: -1 });

        res.status(200).json({
            success: true,
            discounts
        });
    } catch (error) {
        console.error('Error fetching active discounts:', error);
        res.status(500).json({
            success: false,
            message: 'Error fetching active discounts',
            error: error.message
        });
    }
};

/**
 * Create discount (admin only)
 */
const createDiscount = async (req, res) => {
    try {
        const { name, type, discountType, value, description, applicablePlans, startDate, endDate } = req.body;

        if (!name || !type || !discountType || value === undefined) {
            return res.status(400).json({
                success: false,
                message: 'Missing required fields: name, type, discountType, value'
            });
        }

        if (discountType === 'percentage' && (value < 0 || value > 100)) {
            return res.status(400).json({
                success: false,
                message: 'Percentage discount must be between 0 and 100'
            });
        }

        const discount = new Discount({
            name,
            type,
            discountType,
            value,
            description,
            applicablePlans: applicablePlans || [],
            startDate: startDate ? new Date(startDate) : new Date(),
            endDate: endDate ? new Date(endDate) : null
        });

        await discount.save();

        res.status(201).json({
            success: true,
            message: 'Discount created successfully',
            discount
        });
    } catch (error) {
        console.error('Error creating discount:', error);
        res.status(500).json({
            success: false,
            message: 'Error creating discount',
            error: error.message
        });
    }
};

/**
 * Update discount (admin only)
 */
const updateDiscount = async (req, res) => {
    try {
        const { discountId } = req.params;
        const updates = req.body;

        if (updates.value !== undefined && updates.discountType === 'percentage' && (updates.value < 0 || updates.value > 100)) {
            return res.status(400).json({
                success: false,
                message: 'Percentage discount must be between 0 and 100'
            });
        }

        if (updates.startDate) updates.startDate = new Date(updates.startDate);
        if (updates.endDate) updates.endDate = new Date(updates.endDate);

        const discount = await Discount.findByIdAndUpdate(
            discountId,
            { $set: updates },
            { new: true, runValidators: true }
        );

        if (!discount) {
            return res.status(404).json({
                success: false,
                message: 'Discount not found'
            });
        }

        res.status(200).json({
            success: true,
            message: 'Discount updated successfully',
            discount
        });
    } catch (error) {
        console.error('Error updating discount:', error);
        res.status(500).json({
            success: false,
            message: 'Error updating discount',
            error: error.message
        });
    }
};

/**
 * Delete discount (admin only)
 */
const deleteDiscount = async (req, res) => {
    try {
        const { discountId } = req.params;

        const discount = await Discount.findByIdAndDelete(discountId);

        if (!discount) {
            return res.status(404).json({
                success: false,
                message: 'Discount not found'
            });
        }

        res.status(200).json({
            success: true,
            message: 'Discount deleted successfully'
        });
    } catch (error) {
        console.error('Error deleting discount:', error);
        res.status(500).json({
            success: false,
            message: 'Error deleting discount',
            error: error.message
        });
    }
};

module.exports = {
    getAllDiscounts,
    getActiveDiscounts,
    createDiscount,
    updateDiscount,
    deleteDiscount
};

