const User = require('../models/User');
const Task = require('../models/Task');
const GroupTask = require('../models/GroupTask');
const ActivityLog = require('../models/ActivityLog');
const moment = require('moment');
// Robust import: support both default function export and named export
const phoneUtils = require('../utils/phoneUtils');
const normalizeEgyptPhone =
    typeof phoneUtils === 'function'
        ? phoneUtils
        : phoneUtils && typeof phoneUtils.normalizeEgyptPhone === 'function'
            ? phoneUtils.normalizeEgyptPhone
            : null;

// Generate daily summary for all users
const getDailySummary = async (req, res) => {
    try {
        console.log('🔍 Starting daily summary generation...');
        
        // Get all users with phone numbers
        const users = await User.find({ 
            phone: { $exists: true, $ne: null, $ne: '' },
            status: 'active'
        }).select('_id name phone');

        console.log(`📱 Found ${users.length} users with phone numbers:`, users.map(u => ({ id: u._id, name: u.name, phone: u.phone })));

        if (!users || users.length === 0) {
            console.log('❌ No users found with phone numbers');
            return res.status(200).json([]);
        }

        const summaries = [];

        for (const user of users) {
            try {
                console.log(`👤 Generating summary for user: ${user.name} (${user._id})`);
                const summary = await generateUserSummary(user);
                console.log(`📝 Summary for ${user.name}:`, summary);
                if (summary) {
                    summaries.push(summary);
                }
            } catch (error) {
                console.error(`Error generating summary for user ${user._id}:`, error);
                // Continue with other users even if one fails
            }
        }

        console.log(`✅ Generated ${summaries.length} summaries total`);
        res.status(200).json(summaries);
    } catch (error) {
        console.error('Error generating daily summaries:', error);
        res.status(500).json({ 
            error: 'Internal server error',
            message: 'Failed to generate daily summaries'
        });
    }
};

// Generate summary for a single user
const generateUserSummary = async (user) => {
    try {
        // Get today's date range
        const today = moment().startOf('day');
        const tomorrow = moment().add(1, 'day').startOf('day');
        const endOfToday = moment().endOf('day');
        const endOfTomorrow = moment().add(1, 'day').endOf('day');

        // Get today's activities
        const todayActivities = await ActivityLog.find({
            userId: user._id,
            date: {
                $gte: today.toDate(),
                $lte: endOfToday.toDate()
            }
        }).sort({ date: -1 });

        // Get personal tasks from sections
        const Section = require('../models/Sections');
        const userSections = await Section.find({ userId: user._id });
        
        let todayPersonalTasks = [];
        let tomorrowPersonalTasks = [];
        
        userSections.forEach(section => {
            section.tasks.forEach(task => {
                if (task.dueDate) {
                    const taskDueDate = moment(task.dueDate);
                    
                    // Check if task is due today and not done
                    if (taskDueDate.isSameOrAfter(today) && taskDueDate.isSameOrBefore(endOfToday) && !task.isDone) {
                        todayPersonalTasks.push({
                            ...task.toObject(),
                            sectionName: section.name
                        });
                    }
                    
                    // Check if task is due tomorrow and not done
                    if (taskDueDate.isSameOrAfter(tomorrow) && taskDueDate.isSameOrBefore(endOfTomorrow) && !task.isDone) {
                        tomorrowPersonalTasks.push({
                            ...task.toObject(),
                            sectionName: section.name
                        });
                    }
                }
            });
        });

        // Get today's pending group tasks
        const todayPendingGroupTasks = await GroupTask.find({
            assignedTo: user.email,
            dueDate: {
                $gte: today.toDate(),
                $lte: endOfToday.toDate()
            },
            completed: false
        }).populate('group', 'name');

        // Get tomorrow's group tasks (only incomplete ones)
        const tomorrowGroupTasks = await GroupTask.find({
            assignedTo: user.email,
            dueDate: {
                $gte: tomorrow.toDate(),
                $lte: endOfTomorrow.toDate()
            },
            completed: false
        }).populate('group', 'name');

        // Generate the summary message
        const summaryMessage = buildSummaryMessage(
            todayActivities,
            [...todayPersonalTasks, ...todayPendingGroupTasks],
            [...tomorrowPersonalTasks, ...tomorrowGroupTasks]
        );

        if (!normalizeEgyptPhone) {
            console.error(`☎️ Phone normalizer not available; skipping user ${user.name} (${user._id}).`);
            return null;
        }

        const normalizedPhone = normalizeEgyptPhone(user.phone);
        if (!normalizedPhone) {
            console.warn(`☎️ Skipping user ${user.name} (${user._id}) due to invalid phone: '${user.phone}'`);
            return null;
        }
        if (normalizedPhone !== user.phone) {
            console.log(`☎️ Normalized phone for ${user.name}: '${user.phone}' -> '${normalizedPhone}'`);
        }

        return {
            to: normalizedPhone,
            summary: summaryMessage
        };
    } catch (error) {
        console.error(`Error generating summary for user ${user._id}:`, error);
        return null;
    }
};

// Build the summary message in Arabic
const buildSummaryMessage = (activities, todayPendingTasks, tomorrowTasks) => {
    let message = '';

    // Activities section
    if (activities && activities.length > 0) {
        message += 'نشاطك النهاردة:\n';
        activities.forEach(activity => {
            switch (activity.action) {
                case 'task_created':
                    message += `- أضفت تاسك: ${activity.taskTitle || 'مهمة جديدة'}`;
                    if (activity.taskDescription) {
                        message += ` (${activity.taskDescription})`;
                    }
                    message += '\n';
                    break;
                case 'task_completed':
                    message += `- أنجزت تاسك: ${activity.taskTitle || 'مهمة'}`;
                    if (activity.taskDescription) {
                        message += ` (${activity.taskDescription})`;
                    }
                    message += '\n';
                    break;
                case 'group_task_assigned':
                    message += `- اتضاف عليك تاسك في الجروب`;
                    if (activity.groupName) {
                        message += `: ${activity.groupName}`;
                    }
                    message += '\n';
                    break;
                case 'task_shared':
                    message += '- اتعمل مشاركة لتاسك معاك في صفحة الأسّايند\n';
                    break;
                default:
                    if (activity.taskTitle) {
                        message += `- ${activity.taskTitle}`;
                        if (activity.taskDescription) {
                            message += ` (${activity.taskDescription})`;
                        }
                        message += '\n';
                    }
            }
        });
    } else {
        message += 'مفيش أي نشاط ليك النهاردة 😴\nشد حيلك بكرة 💪\n';
    }

    message += '\n';

    // Today's pending tasks section
    if (todayPendingTasks && todayPendingTasks.length > 0) {
        message += 'المهام اللي متبقيالك النهاردة:\n';
        todayPendingTasks.forEach(task => {
            // Handle both personal tasks and group tasks
            const taskTitle = task.title || task.name || 'مهمة بدون عنوان';
            message += `- ${taskTitle}`;
            if (task.description) {
                message += ` (${task.description})`;
            }
            message += '\n';
        });
    } else {
        message += 'مبروك 🎉 خلصت كل مستحقاتك النهاردة ✅\n';
    }

    message += '\n';

    // Tomorrow's tasks section
    if (tomorrowTasks && tomorrowTasks.length > 0) {
        message += 'المهام اللي عليك بكرة:\n';
        tomorrowTasks.forEach(task => {
            // Handle both personal tasks and group tasks
            const taskTitle = task.title || task.name || 'مهمة بدون عنوان';
            message += `- ${taskTitle}`;
            if (task.description) {
                message += ` (${task.description})`;
            }
            message += '\n';
        });
    } else {
        message += 'مفيش مهام ليك بكرة 😎\n';
    }

    // Add website link CTA in Arabic
    message += '\n\nللوصول بسهولة إلى الموقع اضغط هنا: https://task-tree.net/';

    return message.trim();
};

module.exports = {
    getDailySummary
};