const Groq = require('groq-sdk');

class GroqService {
  constructor() {
    const apiKey = process.env.GROQ_API_KEY;
    if (apiKey && String(apiKey).trim()) {
      this.groq = new Groq({ apiKey });
    } else {
      // Gracefully degrade when no API key is present
      this.groq = null;
    }
    
    // Fallback model priority order - primary model first, then openai/gpt-oss-120b as fallback
    this.models = [
      'llama-3.3-70b-versatile',
      'openai/gpt-oss-120b',
      'llama-3.1-8b-instant',
      'meta-llama/llama-4-maverick-17b-128e-instruct',
      'meta-llama/llama-4-scout-17b-16e-instruct',
      'meta-llama/llama-guard-4-12b',
      'meta-llama/llama-prompt-guard-2-22m',
      'meta-llama/llama-prompt-guard-2-86m',
      'moonshotai/kimi-k2-instruct',
      'moonshotai/kimi-k2-instruct-0905',
      'openai/gpt-oss-20b',
      'qwen/qwen3-32b',
      'groq/compound',
      'groq/compound-mini'
    ];
    
    this.currentModelIndex = 0;
    this.model = this.models[this.currentModelIndex];
  }

  // System prompt for Titi's personality and behavior
  getSystemPrompt(userContext = {}) {
    let taskContextInfo = '';
    
    if (userContext.taskContext) {
      const { workspaces, urgentTasks, todayTasks, overdueCount, highPriorityCount } = userContext.taskContext;
      
      // Add workspace and section information
      if (workspaces && workspaces.length > 0) {
        taskContextInfo += '\n\nمساحات العمل والأقسام والمهام الخاصة بالمستخدم:\n';
        workspaces.forEach((workspace, index) => {
          taskContextInfo += `${index + 1}. مساحة العمل "${workspace.name}":\n`;
          if (workspace.sections && workspace.sections.length > 0) {
            workspace.sections.forEach((section, sectionIndex) => {
              taskContextInfo += `   - القسم ${sectionIndex + 1}: ${section.name}\n`;
              if (section.tasks && section.tasks.length > 0) {
                taskContextInfo += `     المهام في هذا القسم:\n`;
                section.tasks.forEach((task, taskIndex) => {
                  const status = task.isDone ? '✅ مكتملة' : '⏳ قيد التنفيذ';
                  const priority = task.priority || 'عادية';
                  const important = task.isImportant ? '⭐ مهمة' : '';
                  taskContextInfo += `       ${taskIndex + 1}. ${task.name} - ${status} - أولوية: ${priority} ${important}\n`;
                  if (task.description) {
                    taskContextInfo += `          الوصف: ${task.description}\n`;
                  }
                  if (task.dueDate) {
                    taskContextInfo += `          تاريخ الاستحقاق: ${new Date(task.dueDate).toLocaleDateString('ar-EG')}\n`;
                  }
                });
              } else {
                taskContextInfo += `     - لا توجد مهام في هذا القسم\n`;
              }
            });
          } else {
            taskContextInfo += `   - لا توجد أقسام في هذه المساحة\n`;
          }
        });
      }
      
      if (urgentTasks && urgentTasks.length > 0) {
        taskContextInfo += '\nالمهام المستعجلة:\n';
        urgentTasks.forEach((task, index) => {
          const status = task.isOverdue ? '⚠️ متأخرة' : '🔥 عالية الأولوية';
          taskContextInfo += `${index + 1}. ${task.name} - ${status}\n`;
        });
      }
      
      if (todayTasks && todayTasks.length > 0) {
        taskContextInfo += `\nمهام اليوم (${todayTasks.length}):\n`;
        todayTasks.forEach((task, index) => {
          taskContextInfo += `${index + 1}. ${task.name} - أولوية ${task.priority}\n`;
        });
      }
      
      if (overdueCount > 0) {
        taskContextInfo += `\n⚠️ عدد المهام المتأخرة: ${overdueCount}\n`;
      }
      
      if (highPriorityCount > 0) {
        taskContextInfo += `🔥 عدد المهام عالية الأولوية: ${highPriorityCount}\n`;
      }
    }

    return `أنت "تيتي"، مساعد ذكي لطيف وودود في نظام TaskTree. شخصيتك:

- اسمك: تيتي 👻
- الشخصية: لطيف، ودود، داعم، مرح قليلاً
- اللغة: العربية (نبرة ودية)
- الأسلوب: محادثة دافئة ومحفزة
- يمكنك استخدام الإيموجي الخفيفة (✅✨💪📅🔥)

مهامك الأساسية:
1. فهم الأوامر الطبيعية بالعربية
2. تنفيذ العمليات في TaskTree (إنشاء workspace، section، task، إلخ)
3. تقديم ملخصات يومية وخطط ذكية
4. التحفيز والدعم
5. تنبيه المستخدم للمهام المستعجلة والمتأخرة
6. الإجابة على أسئلة المستخدم حول مساحات العمل والأقسام الخاصة به

${taskContextInfo}

عند تحليل رسالة المستخدم، حدد النية (intent) من هذه الخيارات:
- create_workspace: إنشاء مساحة عمل جديدة
- create_section: إنشاء قسم جديد
- create_task: إنشاء مهمة جديدة
- update_task: تحديث مهمة موجودة
- get_summary: طلب ملخص يومي
- get_plan: طلب خطة ذكية
- general_chat: محادثة عامة

استجب دائماً بالعربية بطريقة ودية ومحفزة. 

وضع المخطط الصارم (STRICT PLANNER MODE):
- لا تُنشئ خطة أو هيكل متعدد الخطوات إلا إذا طلب المستخدم ذلك صراحة باستخدام كلمات مثل: "خطة", "plan", "برنامج", "جدول", "هيكل".
- عدم إنتاج [ACTIONS] في هذه الحالات يُعد خطأ. لا تكتفِ بـ [INTENT].
- يجب أن تكون [ACTIONS] في صيغة JSON صحيحة بالكامل باستخدام علامات اقتباس مزدوجة فقط.
- استخدم الحقول بالإنجليزية كما هي: intent, params, و dependsOn فقط. لا تُضيف حقولًا أخرى.
- استخدم أسماء الكيانات كما كتبها المستخدم حرفيًا (بدون تعديل أو تعريب إضافي أو تغيير حالة الأحرف).
- رتّب الاعتمادية بدقة: create_workspace أولًا، ثم create_section التابعة له مع dependsOn على فهرس الـ workspace، ثم create_task التابعة للقسم مع dependsOn على فهرس القسم.
- اجعل [ACTIONS] آخر جزء في الرسالة، ولا ترفق أي نص بعدها.
- إذا كانت الخطة طويلة، فأنشئ على الأقل مساحة العمل والأقسام الأساسية ضمن [ACTIONS]، ويمكن إضافة المهام لاحقًا إذا كان الطول عائقًا.
- ملاحظة هامة: لا تُنشئ "خطة دراسية" أو "خطة قواعد بيانات" لمجرد ذكر DB أو قواعد البيانات. اعتبر ذلك محادثة عامة ما لم يطلب المستخدم خطة صراحة.

عندما يسأل المستخدم عن مساحات العمل أو الأقسام، استخدم البيانات الحقيقية المتوفرة أعلاه فقط. لا تخترع أسماء أو معلومات غير موجودة.

إذا كان لدى المستخدم مهام مستعجلة أو متأخرة، اذكرها في ردك وحفزه على إنجازها.

أولاً: إذا كان هناك أمر واحد فقط قابل للتنفيذ، اذكره في نهاية ردك بتنسيق:
[INTENT: نوع_النية]
[PARAMS: المعاملات المطلوبة بتنسيق JSON]

ثانياً: إذا كانت رسالة المستخدم تحتوي على عدة أوامر في وقت واحد، فقم بتحليلها إلى قائمة إجراءات مرتبة (stack)، مع تحديد العلاقات بينها (اعتمادات/ترتيب) حتى يتم تنفيذها بشكل صحيح:

[ACTIONS: [
  {"intent":"create_workspace","params":{"name":"..."}},
  {"intent":"create_section","params":{"name":"...","workspaceName":"..."},"dependsOn":[0]},
  {"intent":"create_task","params":{"name":"...","sectionName":"...","workspaceName":"..."},"dependsOn":[1]}
]]

إرشادات ACTIONS:
 - استخدم الحقول: intent, params فقط، وأضِف dependsOn عند الحاجة.
 - يعتمد الحقل dependsOn على فهارس الإجراءات داخل نفس قائمة ACTIONS (تبدأ من 0). اربط الإجراء التابع بالإجراءات التي تسبقه مباشرة.
  - عندما تكون الأوامر غير مرتبطة ببعضها، اترك dependsOn فارغًا لتسمح بالتنفيذ المتوازي.
  - فضّل استخدام workspaceName و sectionName للإشارة إلى الكيانات الاسمية.
  - لا تستخدم مفاتيح عربية داخل JSON؛ المفاتيح يجب أن تكون بالإنجليزية (intent, params, dependsOn).
  - قاعدة حديدية: عند طلب "خطة" أو "بناء هيكل" أو أي أمر يوحي بإنشاء عدة كيانات (workspace/sections/tasks)، يجب أن تُرجع دائمًا كتلة [ACTIONS] كاملة، ولا تكتفي بـ [INTENT] وحده.
  - حافظ على الأسماء كما كتبها المستخدم حرفيًا بدون أي تعديل أو تعريب أو تحويل حالة الأحرف.
  - إن كانت الاستجابة طويلة جدًا، أنشئ على الأقل مساحة العمل والأقسام الأساسية ضمن [ACTIONS]، ويمكن إضافة المهام لاحقًا.
  - تأكّد من ترتيب dependsOn بشكل صحيح: إنشاء workspace أولاً، ثم الأقسام التابعة لها، ثم المهام التابعة للأقسام.


مثال لخطة كاملة ومفصلة (هيكلة البيانات):
المستخدم: "اعمل خطة كاملة للداتا ستركشر، أقسام أساسية، وكل قسم فيه تاسكات"
الرد: "تمام! سأبني لك خطة مفصلة لهيكلة البيانات مع أقسام أساسية وتاسكات عملية لكل قسم ✨"
[ACTIONS: [
  {"intent":"create_workspace","params":{"name":"Data Structure Study Plan"}},
  {"intent":"create_section","params":{"name":"التعريفات والأساسيات","workspaceName":"Data Structure Study Plan"},"dependsOn":[0]},
  {"intent":"create_section","params":{"name":"Arrays و Linked Lists","workspaceName":"Data Structure Study Plan"},"dependsOn":[0]},
  {"intent":"create_section","params":{"name":"Stacks و Queues","workspaceName":"Data Structure Study Plan"},"dependsOn":[0]},
  {"intent":"create_section","params":{"name":"Trees و Graphs","workspaceName":"Data Structure Study Plan"},"dependsOn":[0]},
  {"intent":"create_section","params":{"name":"الخوارزميات والتقنيات المتقدمة","workspaceName":"Data Structure Study Plan"},"dependsOn":[0]},
  {"intent":"create_task","params":{"name":"قراءة الفصل التمهيدي","description":"مقدمة عن أنواع الهياكل ومجالات استخدامها","priority":"medium","sectionName":"التعريفات والأساسيات","workspaceName":"Data Structure Study Plan"},"dependsOn":[1]},
  {"intent":"create_task","params":{"name":"مشاهدة فيديو تعريفي","description":"فيديو 20 دقيقة عن المفاهيم الأساسية","priority":"low","sectionName":"التعريفات والأساسيات","workspaceName":"Data Structure Study Plan"},"dependsOn":[1]},
  {"intent":"create_task","params":{"name":"حل تمارين الأساسيات","description":"أسئلة اختيار من متعدد وتمارين قصيرة","priority":"medium","sectionName":"التعريفات والأساسيات","workspaceName":"Data Structure Study Plan"},"dependsOn":[1]},
  {"intent":"create_task","params":{"name":"مذاكرة Arrays","description":"أنواع الـ Arrays وعملياتها","priority":"medium","sectionName":"Arrays و Linked Lists","workspaceName":"Data Structure Study Plan"},"dependsOn":[2]},
  {"intent":"create_task","params":{"name":"مذاكرة Linked Lists","description":"Singly/Doubly/Circular Linked Lists","priority":"medium","sectionName":"Arrays و Linked Lists","workspaceName":"Data Structure Study Plan"},"dependsOn":[2]},
  {"intent":"create_task","params":{"name":"حل تمارين Arrays & LL","description":"تطبيقات وإنشاء هياكل بسيطة","priority":"high","sectionName":"Arrays و Linked Lists","workspaceName":"Data Structure Study Plan"},"dependsOn":[2]},
  {"intent":"create_task","params":{"name":"مذاكرة Stacks","description":"المفاهيم والاستخدامات","priority":"medium","sectionName":"Stacks و Queues","workspaceName":"Data Structure Study Plan"},"dependsOn":[3]},
  {"intent":"create_task","params":{"name":"مذاكرة Queues","description":"أنواع الطوابير وعملياتها","priority":"medium","sectionName":"Stacks و Queues","workspaceName":"Data Structure Study Plan"},"dependsOn":[3]},
  {"intent":"create_task","params":{"name":"حل تمارين Stacks & Queues","description":"مسائل تطبيقية","priority":"high","sectionName":"Stacks و Queues","workspaceName":"Data Structure Study Plan"},"dependsOn":[3]},
  {"intent":"create_task","params":{"name":"مذاكرة Trees","description":"Binary Trees وعملياتها","priority":"medium","sectionName":"Trees و Graphs","workspaceName":"Data Structure Study Plan"},"dependsOn":[4]},
  {"intent":"create_task","params":{"name":"مذاكرة Graphs","description":"تمثيل الرسوم وخوارزميات البحث","priority":"medium","sectionName":"Trees و Graphs","workspaceName":"Data Structure Study Plan"},"dependsOn":[4]},
  {"intent":"create_task","params":{"name":"حل تمارين Trees & Graphs","description":"Traversal، أقصر مسار، وغيرها","priority":"high","sectionName":"Trees و Graphs","workspaceName":"Data Structure Study Plan"},"dependsOn":[4]},
  {"intent":"create_task","params":{"name":"الخوارزميات الأساسية","description":"Time/Space Complexity، Sorting، Searching","priority":"medium","sectionName":"الخوارزميات والتقنيات المتقدمة","workspaceName":"Data Structure Study Plan"},"dependsOn":[5]},
  {"intent":"create_task","params":{"name":"تقنيات متقدمة","description":"Dynamic Programming، Greedy، Divide & Conquer","priority":"medium","sectionName":"الخوارزميات والتقنيات المتقدمة","workspaceName":"Data Structure Study Plan"},"dependsOn":[5]},
  {"intent":"create_task","params":{"name":"حل تمارين الخوارزميات","description":"مسائل متنوعة بمستويات متعددة","priority":"high","sectionName":"الخوارزميات والتقنيات المتقدمة","workspaceName":"Data Structure Study Plan"},"dependsOn":[5]}
]]


مثال:
المستخدم: "أنشالي workspace اسمها Study Plan"
الرد: "ممتاز! سأنشئ لك مساحة عمل جديدة باسم 'Study Plan' ✨ هذا سيساعدك على تنظيم دراستك بشكل رائع! 📚
[INTENT: create_workspace]
[PARAMS: {"name": "Study Plan"}]"

مثال لإنشاء قسم:
المستخدم: "قم بانشاء سكشن اسمه تيتي في ووركسبيس تيتي"
الرد: "ممتاز! سأنشئ لك قسمًا جديدًا باسم 'تيتي' داخل مساحة العمل 'تيتي' ✨ سيساعدك هذا القسم على تنظيم مهامك بطريقة أكثر وضوحًا وسهولة! 💪
[INTENT: create_section]
[PARAMS: {"name": "تيتي", "workspaceName": "تيتي"}]"

مثال آخر لإنشاء مهمة:
المستخدم: "ممكن تعملي تاسك في الووركسبيس هوم ف السكشن هوم ويكون اسمها ازيك ياحمد"
الرد: "ممتاز! سأنشئ لك مهمة جديدة باسم 'ازيك ياحمد' في القسم 'هوم' في مساحة العمل 'هوم' ✨
[INTENT: create_task]
[PARAMS: {"name": "ازيك ياحمد", "sectionName": "Home", "workspaceName": "Home"}]"

مثال متعدد الأوامر (ترتيب متسلسل):
المستخدم: "اعمل workspace اسمها serial وبعدين جواها سكشن اسمه serial وبعدين جوه السكشن اعمل تاسك اسمها hola"
الرد: "تمام! هننشئ مساحة عمل اسمها 'serial' ثم قسم اسمه 'serial' داخلها، وأخيراً مهمة اسمها 'hola' ✨"
[ACTIONS: [
  {"intent":"create_workspace","params":{"name":"serial"}},
  {"intent":"create_section","params":{"name":"serial","workspaceName":"serial"},"dependsOn":[0]},
  {"intent":"create_task","params":{"name":"hola","sectionName":"serial","workspaceName":"serial"},"dependsOn":[1]}
]]

مهم جدًا:
- عند ظهور كلمات مثل "خطة", "plan", "بروجرام", "جدول", "سكاشن كتير", اعتبر الطلب متعدد الخطوات وأنتج [ACTIONS] كاملة.
- لا تغيّر أسماء الأقسام أو المساحات أو المهام؛ استخدم النص كما ورد من المستخدم بالضبط.
- إن لم تتوفر معلومة لازمة لإجراء معين، ضعه في [ACTIONS] وسيُعالج التنفيذ اعتماداته تلقائيًا أو يُعلَم المستخدم بالنقص.
`;
  }

  // Generate AI response using Groq
  async generateResponse(userMessage, chatHistory = [], userContext = {}) {
    let lastError = null;
    
    // Try each model in order until one succeeds
    for (let modelIndex = this.currentModelIndex; modelIndex < this.models.length; modelIndex++) {
      const currentModel = this.models[modelIndex];
      
      try {
        console.log(`🤖 Trying model: ${currentModel}`);
        
        // Prepare conversation history
        const messages = [
          {
            role: 'system',
            content: this.getSystemPrompt(userContext)
          }
        ];

        // Add recent chat history for context
        if (chatHistory && chatHistory.length > 0) {
          chatHistory.forEach(msg => {
            messages.push({
              role: msg.sender === 'user' ? 'user' : 'assistant',
              content: msg.text
            });
          });
        }

        // Add current user message
        messages.push({
          role: 'user',
          content: userMessage
        });

        // Call Groq API with current model
        const completion = await this.groq.chat.completions.create({
          messages,
          model: currentModel,
          temperature: 0.7,
          max_tokens: 1000,
          top_p: 1,
          stream: false
        });

        const response = completion.choices[0]?.message?.content || 'عذراً، لم أستطع فهم طلبك. يمكنك إعادة المحاولة؟ 😊';

        // Parse intent(s) and parameters from response
        const { cleanResponse, intent, params, actions } = this.parseIntentFromResponse(response);

        // If user explicitly asked for a plan/structure but model didn't include ACTIONS,
        // derive a silent ACTIONS list from the user message (plan-only, no auto plans).
        let derivedActions = actions;
        const mentionsSection = /(سكشن|قسم|section)/i.test(userMessage);
        const mentionsTask = /(تاسك|مهمة|task)/i.test(userMessage);
        const mentionsWorkspace = /(workspace|مساحة\s*عمل)/i.test(userMessage);
        const planCue = /(خطة|plan|برنامج|جدول|هيكل)/i.test(userMessage);
        const verbCue = /(اعمل|اعملي|أنشئ|انشاء|إنشاء)/i.test(userMessage);
        const multiTypeMention = [mentionsSection, mentionsTask, mentionsWorkspace].filter(Boolean).length >= 2;
        const multiOpsCue = verbCue && multiTypeMention; // stricter: at least two entity types mentioned
        if ((!derivedActions || derivedActions.length === 0) && planCue) {
          try {
            const offlineActions = this.generateOfflinePlanActions(userMessage);
            if (offlineActions && offlineActions.length > 0) {
              derivedActions = offlineActions;
            }
          } catch {}
        }

        console.log(`✅ Success with model: ${currentModel}`);
        
        // Update current model index for future requests
        this.currentModelIndex = modelIndex;
        this.model = currentModel;

        return {
          response: cleanResponse,
          intent,
          params,
          actions: Array.isArray(derivedActions) ? derivedActions : null,
          rawResponse: response,
          modelUsed: currentModel
        };

      } catch (error) {
        lastError = error;
        console.error(`❌ Error with model ${this.models[modelIndex]}:`, error.message);
        
        // Check if it's a rate limit error
        if (error.status === 429 || error.message.includes('rate_limit_exceeded')) {
          console.log(`⏭️ Rate limit reached for ${this.models[modelIndex]}, trying next model...`);
          continue; // Try next model
        } else {
          // For other errors, also try next model but log differently
          console.log(`⚠️ API error with ${this.models[modelIndex]}, trying next model...`);
          continue;
        }
      }
    }

    // If all models failed, try an offline planner fallback for strict planning requests
    console.error('💥 All models failed. Last error:', lastError);

    const offlinePlanActions = this.generateOfflinePlanActions(userMessage);
    if (offlinePlanActions && offlinePlanActions.length > 0) {
      const wsAction = offlinePlanActions.find(a => a.intent === 'create_workspace');
      const wsName = wsAction?.params?.name || 'Study Plan';
      const sectionNames = offlinePlanActions
        .filter(a => a.intent === 'create_section')
        .map(a => a.params?.name)
        .filter(Boolean)
        .slice(0, 12);

      const friendlyMsg = `تمام! فيه عطل بسيط في النموذج، فجهزت لك خطة منظمة بدون أي JSON:

- مساحة العمل: "${wsName}"
- الأقسام:
${sectionNames.map(n => `  • ${n}`).join('\n')}

هضيف ٣ مهام تدريبية لكل قسم (قراءة، تمرين عملي، مراجعة سريعة). لو كل شيء تمام، هنفّذ الخطة فوراً ✅`;

      return {
        response: friendlyMsg,
        intent: 'multi',
        params: null,
        actions: offlinePlanActions,
        error: lastError?.message || 'All models failed',
        allModelsFailed: true
      };
    }

    // Otherwise, return technical error response
    return {
      response: 'عذراً، حدث خطأ تقني. سأكون معك قريباً! 🔧',
      intent: null,
      params: null,
      actions: null,
      error: lastError?.message || 'All models failed',
      allModelsFailed: true
    };
  }

  // Generate a short motivational message (Arabic, Egyptian dialect) with a single quoted saying and author
  // Returns plain text like:
  // "رسالة قصيرة تحفيزية …\n\n“المقولة” — اسم المؤلف"
  async generateMotivation(user = {}) {
    // Ensure API key exists; otherwise let caller decide fallback
    if (!process.env.GROQ_API_KEY) {
      throw new Error('Missing GROQ_API_KEY');
    }

    let lastError = null;
    for (let modelIndex = this.currentModelIndex; modelIndex < this.models.length; modelIndex++) {
      const currentModel = this.models[modelIndex];
      try {
        const systemPrompt = [
          'أنت "تيتي"، مساعد لطيف وودود.',
          'اكتب رسالة قصيرة باللهجة المصرية، مشجعة وبسيطة، لا تتجاوز سطرين.',
          'يجب أن تحتوي الرسالة على "مقولة واحدة" بين علامات اقتباس “ ... ” أو "..."،',
          'وفي النهاية أضف اسم صاحب المقولة مسبوقًا بشرطة طويلة هكذا: — اسم المؤلف.',
          'نوّع المؤلفين (ألبرت أينشتاين، ماري كوري، إسحاق نيوتن، ابن سينا، ابن الهيثم، ابن خلدون، جلال الدين الرومي، هنري فورد، بيتر دركر، توماس إديسون، الخوارزمي، طه حسين، ليوناردو دافنشي...).',
          'تجنّب استخدام نابليون بونابرت كاختيار افتراضي، ولا تكرر نفس المؤلف في نفس اليوم إن أمكن.',
          'استخدم نبرة محفزة وودودة مع اسم المستخدم لو متاح.',
          'أعد نصًا فقط بدون JSON أو وسوم أو كود أو [INTENT] أو [ACTIONS].',
          'الحد الأقصى للطول 220 حرفًا.'
        ].join(' ');

        const userContextLines = [
          `الاسم: ${user.name || 'صديق تيتي'}`,
          `نشاط اليوم: ${user.activeToday ? 'نشيط' : 'غير نشيط'}`,
          `تغيّر التقدم: ${user.progressChange || 'غير متوفر'}`,
          'اكتب الرسالة الآن.'
        ].join('\n');

        const messages = [
          { role: 'system', content: systemPrompt },
          { role: 'user', content: userContextLines }
        ];

        const completion = await this.groq.chat.completions.create({
          messages,
          model: currentModel,
          temperature: 0.85,
          max_tokens: 220,
          top_p: 1,
          stream: false
        });

        const text = completion.choices[0]?.message?.content?.trim();
        if (text && text.length > 0) {
          // Record successful model for future calls
          this.currentModelIndex = modelIndex;
          this.model = currentModel;
          return text;
        }
        throw new Error('Empty Groq response');
      } catch (error) {
        lastError = error;
        const msg = String(error?.message || '').toLowerCase();
        if (error.status === 429 || msg.includes('rate') || msg.includes('limit')) {
          // Try next model
          continue;
        } else {
          // Try next model as well for robustness
          continue;
        }
      }
    }
    throw lastError || new Error('All Groq models failed');
  }

  // Parse intent and parameters from AI response
  parseIntentFromResponse(response) {
    const normalizeJson = (txt) => {
      return (txt || '')
        .replace(/[“”]/g, '"')
        .replace(/[‘’]/g, '"')
        .replace(/`/g, '"')
        .replace(/\r?\n/g, ' ')
        .replace(/\t/g, ' ')
        .replace(/\s{2,}/g, ' ')
        .replace(/'\s*:/g, '" :')
        .replace(/:\s*'/g, ': "')
        .replace(/'\s*,/g, '" ,')
        .replace(/,\s*([}\]])/g, '$1')
        .replace(/\/\/.*$/mg, '')
        .replace(/\/\*[\s\S]*?\*\//g, '');
    };
    const intentMatch = response.match(/\[INTENT:\s*([^\]]+)\]/);
    const paramsMatch = response.match(/\[PARAMS:\s*({[\s\S]*?})\]/);
    const actionsMatch = response.match(/\[ACTIONS:\s*(\[[\s\S]*?\])\]/);

    let intent = null;
    let params = null;
    let actions = null;

    if (intentMatch) {
      intent = intentMatch[1].trim();
    }

    const scanAndParseActions = () => {
      try {
        const tokenIdx = response.indexOf('[ACTIONS:');
        const firstArrayStartRel = response.slice(tokenIdx).indexOf('[');
        if (firstArrayStartRel !== -1) {
          const startIdx = tokenIdx + firstArrayStartRel; // index of '[' starting the array
          let bracketDepth = 0;
          let braceDepth = 0;
          let endIdx = -1;
          for (let i = startIdx; i < response.length; i++) {
            const ch = response[i];
            if (ch === '[') bracketDepth++;
            else if (ch === ']') {
              bracketDepth--;
              if (bracketDepth === 0 && braceDepth === 0) { endIdx = i; break; }
            } else if (ch === '{') {
              braceDepth++;
            } else if (ch === '}') {
              braceDepth--;
            }
          }
          let rawBlock;
          if (endIdx !== -1) {
            rawBlock = response.slice(startIdx, endIdx + 1);
          } else {
            // Salvage by closing brackets based on remaining depth
            rawBlock = response.slice(startIdx) + '}'.repeat(Math.max(braceDepth, 0)) + ']'.repeat(Math.max(bracketDepth, 1));
          }
          const sanitized = normalizeJson(rawBlock).replace(/'/g, '"');
          const candidate = JSON.parse(sanitized);
          if (Array.isArray(candidate) && candidate.every(it => typeof it === 'object' && it && 'intent' in it)) {
            return candidate;
          }
        }
      } catch (e) {
        console.error('Bracket-scan extraction of ACTIONS failed:', e);
      }
      return null;
    };

    if (actionsMatch) {
      const rawActions = actionsMatch[1];
      try {
        actions = JSON.parse(normalizeJson(rawActions));
      } catch (e) {
        console.error('Error parsing actions:', e);
        // Fallback: try to sanitize common non-standard JSON issues
        try {
          const sanitized = normalizeJson(rawActions).replace(/'/g, '"');
          actions = JSON.parse(sanitized);
        } catch (e2) {
          console.error('Fallback parsing actions failed:', e2);
          // As a last resort, try bracket-depth scan recovery
          if (response.includes('[ACTIONS:')) {
            const recovered = scanAndParseActions();
            if (recovered) actions = recovered;
            // If still null, attempt object-by-object extraction inside the block
            if (!actions) {
              try {
                const tokenIdx = response.indexOf('[ACTIONS:');
                const arrStartRel = response.slice(tokenIdx).indexOf('[');
                const startIdx = tokenIdx + arrStartRel + 1; // first char after '['
                let i = startIdx;
                const extracted = [];
                let braceDepth = 0;
                let bracketDepth = 1; // we are inside the outer array already
                let objStart = -1;
                for (; i < response.length; i++) {
                  const ch = response[i];
                  if (objStart === -1) {
                    if (ch === '{') { objStart = i; braceDepth = 1; }
                  } else {
                    if (ch === '{') braceDepth++;
                    else if (ch === '}') {
                      braceDepth--;
                      if (braceDepth === 0) {
                        const objRaw = response.slice(objStart, i + 1);
                        objStart = -1;
                        try {
                          const objSan = normalizeJson(objRaw).replace(/'/g, '"');
                          const parsed = JSON.parse(objSan);
                          if (parsed && typeof parsed === 'object' && 'intent' in parsed) {
                            extracted.push(parsed);
                          }
                        } catch {}
                      }
                    }
                  }
                  if (ch === '[') bracketDepth++;
                  else if (ch === ']') {
                    bracketDepth--;
                    if (bracketDepth === 0) break; // end of actions array
                  }
                }
                if (extracted.length > 0) actions = extracted;
              } catch (e3) {
                console.error('Object-by-object ACTIONS extraction failed:', e3);
              }
            }
          }
        }
      }
    } else if (response.includes('[ACTIONS:')) {
      // Attempt to extract a possibly truncated ACTIONS array using bracket-depth scanning
      const recovered = scanAndParseActions();
      if (recovered) actions = recovered;
      else {
        // Attempt object-by-object extraction
        try {
          const tokenIdx = response.indexOf('[ACTIONS:');
          const arrStartRel = response.slice(tokenIdx).indexOf('[');
          const startIdx = tokenIdx + arrStartRel + 1;
          let i = startIdx;
          const extracted = [];
          let braceDepth = 0;
          let bracketDepth = 1;
          let objStart = -1;
          for (; i < response.length; i++) {
            const ch = response[i];
            if (objStart === -1) {
              if (ch === '{') { objStart = i; braceDepth = 1; }
            } else {
              if (ch === '{') braceDepth++;
              else if (ch === '}') {
                braceDepth--;
                if (braceDepth === 0) {
                  const objRaw = response.slice(objStart, i + 1);
                  objStart = -1;
                  try {
                    const objSan = normalizeJson(objRaw).replace(/'/g, '"');
                    const parsed = JSON.parse(objSan);
                    if (parsed && typeof parsed === 'object' && 'intent' in parsed) {
                      extracted.push(parsed);
                    }
                  } catch {}
                }
              }
            }
            if (ch === '[') bracketDepth++;
            else if (ch === ']') {
              bracketDepth--;
              if (bracketDepth === 0) break;
            }
          }
          if (extracted.length > 0) actions = extracted;
        } catch (e3) {
          console.error('Object-by-object ACTIONS extraction failed:', e3);
        }
      }
    }

    // If parsed actions are clearly not action objects (e.g., [0]), try bracket-scan recovery
    if (Array.isArray(actions) && actions.length > 0 && (typeof actions[0] !== 'object' || !('intent' in actions[0]))) {
      try {
        const tokenIdx = response.indexOf('[ACTIONS:');
        if (tokenIdx !== -1) {
          const firstArrayStartRel = response.slice(tokenIdx).indexOf('[');
          if (firstArrayStartRel !== -1) {
            const startIdx = tokenIdx + firstArrayStartRel;
            let depth = 0;
            let endIdx = -1;
            for (let i = startIdx; i < response.length; i++) {
              const ch = response[i];
              if (ch === '[') depth++;
              else if (ch === ']') {
                depth--;
                if (depth === 0) { endIdx = i; break; }
              }
            }
            let rawBlock = endIdx !== -1 ? response.slice(startIdx, endIdx + 1) : response.slice(startIdx) + ']'.repeat(Math.max(depth, 1));
            const sanitized = normalizeJson(rawBlock).replace(/'/g, '"');
            const candidate = JSON.parse(sanitized);
            if (Array.isArray(candidate) && candidate.every(it => typeof it === 'object' && it && 'intent' in it)) {
              actions = candidate;
            }
          }
        }
      } catch (e) {
        console.error('Recovery parse of ACTIONS failed:', e);
      }
    }

    if (paramsMatch) {
      try {
        params = JSON.parse(paramsMatch[1]);
      } catch (e) {
        console.error('Error parsing params:', e);
      }
    }

    // Fallback intent detection if no explicit intent found
    if (!intent) {
      console.log('🔍 No explicit intent found, trying fallback detection...');
      
      // Check for simple greetings first
      const greetingPatterns = [
        /^(hi|hello|hey|مرحبا|أهلا|السلام عليكم|صباح الخير|مساء الخير)$/i,
        /^(how are you|كيف حالك|إزيك|ازيك)$/i
      ];
      
      const isSimpleGreeting = greetingPatterns.some(pattern => pattern.test(response.trim()));
      if (isSimpleGreeting) {
        intent = 'general_chat';
        params = {};
      } else {
        // Try to detect intents from Arabic text patterns
        const lowerResponse = response.toLowerCase();
        
        // Create section patterns
        if (lowerResponse.includes('سكشن') || lowerResponse.includes('قسم')) {
          if (lowerResponse.includes('انشاء') || lowerResponse.includes('إنشاء') || 
              lowerResponse.includes('اعمل') || lowerResponse.includes('اعملي') ||
              lowerResponse.includes('جديد') || lowerResponse.includes('create')) {
            intent = 'create_section';
            
            // Try to extract section name and workspace
            const sectionNameMatch = response.match(/سكشن\s+اسمه\s+([^\s]+)|قسم\s+اسمه\s+([^\s]+)|سكشن\s+([^\s]+)|قسم\s+([^\s]+)/i);
            const workspaceMatch = response.match(/ووركسبيس\s+([^\s]+)|workspace\s+([^\s]+)|في\s+([^\s]+)/i);
            
            params = {};
            if (sectionNameMatch) {
              params.name = sectionNameMatch[1] || sectionNameMatch[2] || sectionNameMatch[3] || sectionNameMatch[4];
            }
            if (workspaceMatch) {
              params.workspaceName = workspaceMatch[1] || workspaceMatch[2] || workspaceMatch[3];
            }
          }
        }
        
        // Create workspace patterns
        else if (lowerResponse.includes('workspace') || lowerResponse.includes('مساحة عمل')) {
          if (lowerResponse.includes('انشاء') || lowerResponse.includes('إنشاء') || 
              lowerResponse.includes('اعمل') || lowerResponse.includes('اعملي') ||
              lowerResponse.includes('جديد') || lowerResponse.includes('create')) {
            intent = 'create_workspace';
            
            const nameMatch = response.match(/workspace\s+اسمها\s+([^\s]+)|اسمها\s+([^\s]+)/i);
            params = {};
            if (nameMatch) {
              params.name = nameMatch[1] || nameMatch[2];
            }
          }
        }
        
        // Create task patterns
        else if (lowerResponse.includes('تاسك') || lowerResponse.includes('مهمة')) {
          if (lowerResponse.includes('انشاء') || lowerResponse.includes('إنشاء') || 
              lowerResponse.includes('اعمل') || lowerResponse.includes('اعملي') ||
              lowerResponse.includes('جديد') || lowerResponse.includes('create')) {
            intent = 'create_task';
            
            const taskNameMatch = response.match(/تاسك\s+اسمها\s+([^\s]+)|مهمة\s+اسمها\s+([^\s]+)|اسمها\s+([^\s]+)/i);
            const sectionMatch = response.match(/سكشن\s+([^\s]+)|قسم\s+([^\s]+)/i);
            const workspaceMatch = response.match(/ووركسبيس\s+([^\s]+)|workspace\s+([^\s]+)/i);
            
            params = {};
            if (taskNameMatch) {
              params.name = taskNameMatch[1] || taskNameMatch[2] || taskNameMatch[3];
            }
            if (sectionMatch) {
              params.sectionName = sectionMatch[1] || sectionMatch[2];
            }
            if (workspaceMatch) {
              params.workspaceName = workspaceMatch[1] || workspaceMatch[2];
            }
          }
        }
        
        // Summary and plan patterns
        else if (lowerResponse.includes('ملخص') || lowerResponse.includes('summary')) {
          intent = 'get_summary';
          params = {};
        }
        else if (lowerResponse.includes('خطة') || lowerResponse.includes('plan')) {
          intent = 'get_plan';
          params = {};
        }
        else if (lowerResponse.includes('مستعجل') || lowerResponse.includes('urgent')) {
          intent = 'urgent_tasks';
          params = {};
        }
      }
      
      if (intent) {
        console.log(`✅ Fallback detection found intent: ${intent}`, params);
      }
    }

    // Remove intent/params/actions from the clean response
    const cleanResponse = response
      .replace(/\[INTENT:\s*[^\]]+\]/g, '')
      .replace(/\[PARAMS:\s*\{[\s\S]*?\}\]/g, '')
      .replace(/\[ACTIONS:\s*\[[\s\S]*?\]\]/g, '')
      .trim();

    return { cleanResponse, intent, params, actions };
  }

  // Offline planner: derive ACTIONS from user message when AI models fail
  generateOfflinePlanActions(userMessage = '') {
    try {
      const msg = (userMessage || '').toString();
      const lower = msg.toLowerCase();
      const hasPlanCue = /(خطة|plan|برنامج|جدول|هيكل)/i.test(msg);
      // Only run offline planner on explicit plan/structure requests
      if (!hasPlanCue) return null;

      // Extract quoted phrases; first one is likely the workspace name
      const quotes = [];
      const quoteRegex = /"([^"]{1,120})"/g;
      let m;
      while ((m = quoteRegex.exec(msg)) !== null) {
        quotes.push(m[1].trim());
      }

      // Workspace name: prefer the first quoted phrase after words indicating title/name
      let workspaceName = null;
      if (quotes.length > 0) workspaceName = quotes[0];
      // Heuristics: look for بعد كلمة (بـعنوان|اسمها|اسم) ثم أول اقتباس
      const titleIdx = msg.search(/(ب?عنوان|اسمها|اسم)/i);
      if (titleIdx !== -1) {
        const afterTitle = msg.slice(titleIdx);
        const afterQuoteMatch = afterTitle.match(/"([^"]{1,120})"/);
        if (afterQuoteMatch) workspaceName = afterQuoteMatch[1].trim();
      }
      // Domain-aware default: DB study plan cues (only when plan requested explicitly)
      const isDB = /(DB|database|قواعد\s*البيانات)/i.test(msg);
      const isPhilosophy = /(الفلسفة|فلسفة|الفلسلة|فلسلة|philosophy)/i.test(msg);
      if (!workspaceName) workspaceName = isDB ? 'DB Study Plan' : 'Study Plan';
      if (!workspaceName && isPhilosophy) workspaceName = 'خطة مذاكرة الفلسفة';

      // Section names: other quoted phrases likely represent sections; also parse after "سكاشن:" list
      let sections = [];
      if (quotes.length > 1) {
        sections = quotes.slice(1);
      }
      const afterSectionsLabel = msg.split(/سكاشن\s*:\s*|sections\s*:\s*/i)[1];
      if (afterSectionsLabel) {
        const secQuotes = [];
        const r = /"([^"]{1,120})"/g;
        let sm;
        while ((sm = r.exec(afterSectionsLabel)) !== null) {
          secQuotes.push(sm[1].trim());
        }
        if (secQuotes.length > 0) sections = secQuotes;
      }
      // Deduplicate sections and keep reasonable limit
      sections = Array.from(new Set(sections)).slice(0, 12);
      if (sections.length === 0) {
        // Provide sensible defaults if none parsed and user asked for a plan
        if (isDB) {
          sections = [
            'مقدمة في قواعد البيانات',
            'نمذجة البيانات',
            'SQL الأساسيات',
            'SQL المتقدم',
            'الفهارس والأداء',
            'المعاملات والتزامن',
            'التطبيع وإلغاء التطبيع',
            'NoSQL الأساسيات',
            'تصميم قواعد البيانات',
            'مشروع عملي صغير',
            'تقييم ومراجعة'
          ];
        } else if (isPhilosophy) {
          sections = [
            'مدخل إلى الفلسفة',
            'الفلسفة اليونانية',
            'الفلسفة الإسلامية',
            'الفلسفة الحديثة والمعاصرة',
            'المنطق ومناهج التفكير',
            'أسئلة وتطبيقات الامتحان',
            'مراجعة عامة'
          ];
        } else {
          sections = ['Indexing & Optimization', 'SQL Practice', 'NoSQL Basics', 'Transactions & Concurrency'];
        }
      }

      // Build actions: workspace, sections, and 3 tasks per section (generic names)
      const actions = [];
      actions.push({ intent: 'create_workspace', params: { name: workspaceName } });

      const sectionIndices = {};
      sections.forEach((secName, idx) => {
        actions.push({ intent: 'create_section', params: { name: secName, workspaceName }, dependsOn: [0] });
        sectionIndices[secName] = actions.length - 1; // index of the section action
      });

      // If the user explicitly asked for tasks in each section, add 3 generic tasks per section
      // For planning cues, default to adding 3 generic tasks per section, support Arabic-Indic digits and common phrasing
      const wantsTasks = (
        hasPlanCue ||
        /(تاسك|task|مهام|مهمة|كل\s*سيكشن\s*فيه)/i.test(msg) ||
        /([0-9\u0660-\u0669]+\s*مهام|ثلاث\s*مهام|3\s*مهام|٣\s*مهام)/i.test(msg)
      );
      if (wantsTasks) {
        const dbTaskSets = {
          'مقدمة في قواعد البيانات': ['لماذا قواعد البيانات؟', 'مصطلحات أساسية (Tables/Rows/Columns)', 'اختبار سريع للمفاهيم'],
          'نمذجة البيانات': ['ERD: الكيانات والعلاقات', 'Cardinality وقيود المفاتيح', 'تحويل ERD إلى مخطط'],
          'SQL الأساسيات': ['SELECT/WHERE/ORDER BY', 'INSERT/UPDATE/DELETE', 'GROUP BY وتمارين التجميع'],
          'SQL المتقدم': ['JOINs المتقدم', 'Subqueries و CTE', 'Views و Procedures الأساسيات'],
          'الفهارس والأداء': ['أنواع الفهارس (B-Tree/Hash)', 'Explain/Analyze وتحسين الاستعلامات', 'قياس الأداء'],
          'المعاملات والتزامن': ['ACID ومستويات العزل', 'Locks و Deadlocks', 'تجارب معاملات متعددة'],
          'التطبيع وإلغاء التطبيع': ['النماذج 1NF–3NF', 'متى نلجأ لإلغاء التطبيع؟', 'تصميم مخطط متوازن'],
          'NoSQL الأساسيات': ['أنواع NoSQL', 'MongoDB CRUD و فهارس', 'اختيار NoSQL مناسب'],
          'تصميم قواعد البيانات': ['جمع المتطلبات', 'اختيار التقنية المناسبة', 'توثيق المخطط ومراجعة'],
          'مشروع عملي صغير': ['تحديد نطاق وكيانات المشروع', 'تنفيذ المخطط وتهيئة البيانات', 'واجهات CRUD واختبار'],
          'تقييم ومراجعة': ['مراجعة المفاهيم', 'تحليل الأداء والملاحظات', 'خطة تطوير لاحقة']
        };

        sections.forEach((secName) => {
          const secIdx = sectionIndices[secName];
          let taskNames = ['Task 1', 'Task 2', 'Task 3'];
          if (isDB && dbTaskSets[secName]) {
            taskNames = dbTaskSets[secName];
          } else if (isPhilosophy) {
            taskNames = ['قراءة أساسية', 'ملخصات ونقاط رئيسية', 'تدريبات وأسئلة'];
          } else if (/sql/i.test(secName)) {
            taskNames = ['SELECT/WHERE basics', 'JOINs practice', 'Aggregation & GROUP BY'];
          } else if (/nosql/i.test(secName)) {
            taskNames = ['NoSQL types overview', 'CRUD in MongoDB', 'Indexing in NoSQL'];
          } else if (/index|performance|الأداء|الفهارس/i.test(secName)) {
            taskNames = ['Index types', 'Query optimization', 'Explain/Analyze'];
          } else if (/transaction|concurrency|المعاملات|التزامن/i.test(secName)) {
            taskNames = ['ACID concepts', 'Isolation levels', 'Deadlocks & locks'];
          }
          taskNames.forEach((tName) => {
            actions.push({ intent: 'create_task', params: { name: tName, sectionName: secName, workspaceName }, dependsOn: [secIdx] });
          });
        });
      }

      return actions;
    } catch (e) {
      console.error('Offline planner failed:', e);
      return null;
    }
  }

  // Generate greeting based on time and user name
  generateGreeting(userName, currentHour = new Date().getHours()) {
    const greetings = {
      morning: [
        `صباح النشاط يا ${userName} ☀️، تيتي هنا يساعدك تنجز يومك بسهولة!`,
        `صباح الخير يا ${userName} 🌅، جاهز لنبدأ يوم مليان إنجاز؟`,
        `أهلاً ${userName}! صباح جميل ومليان طاقة 💪`
      ],
      afternoon: [
        `أهلاً ${userName} 🌞، إيه أخبار الإنجازات النهارده؟`,
        `مرحباً يا ${userName}، تيتي هنا يساعدك تكمل يومك بقوة! 🔥`,
        `أهلاً وسهلاً ${userName}! خلينا نشوف إيه اللي محتاجين ننجزه 📋`
      ],
      evening: [
        `مساء الخير يا ${userName} 🌙، خلينا نراجع إنجازات النهارده!`,
        `مساء النور ${userName}! وقت مراجعة المهام وتخطيط بكرة ✨`,
        `أهلاً ${userName}، مساء هادئ ومنتج 🌆`
      ]
    };

    let timeOfDay;
    if (currentHour >= 5 && currentHour < 12) {
      timeOfDay = 'morning';
    } else if (currentHour >= 12 && currentHour < 18) {
      timeOfDay = 'afternoon';
    } else {
      timeOfDay = 'evening';
    }

    const greetingOptions = greetings[timeOfDay];
    return greetingOptions[Math.floor(Math.random() * greetingOptions.length)];
  }

  // Generate daily summary prompt
  generateSummaryPrompt(tasksData) {
    return `قم بتحليل مهام اليوم وإعطاء ملخص ودود بالعربية:

المهام المستحقة اليوم: ${tasksData.todayTasks || 0}
المهام المتأخرة: ${tasksData.overdueTasks || 0}
المهام المكتملة مؤخراً: ${tasksData.completedTasks || 0}
المهام عالية الأولوية: ${tasksData.highPriorityTasks || 0}

اكتب ملخص محفز وودود بشخصية تيتي، مع اقتراحات للتحسين.`;
  }

  // Generate smart plan prompt
  generatePlanPrompt(tasksData) {
    return `بناءً على المهام التالية، اقترح خطة عمل ذكية بالعربية:

${JSON.stringify(tasksData, null, 2)}

رتب المهام حسب الأولوية والمواعيد النهائية، واقترح أفضل ترتيب للعمل بشخصية تيتي المحفزة.`;
  }
}

module.exports = new GroqService();