const mongoose = require('mongoose');
const dotenv = require('dotenv');
const moment = require('moment');
const path = require('path');

// Load env from server directory
dotenv.config({ path: path.join(__dirname, '../.env') });

const User = require('../models/User');
const Sections = require('../models/Sections');
const GroupTask = require('../models/GroupTask');
const Event = require('../models/Event');
const GroupMembership = require('../models/GroupMembership');

const connectDB = async () => {
    try {
        await mongoose.connect(process.env.MONGO_URI);
        console.log('MongoDB Connected');
    } catch (err) {
        console.error('MongoDB Connection Error:', err.message);
        process.exit(1);
    }
};

const debugUser = async () => {
    await connectDB();

    const email = 'nadyahmd88@gmail.com';
    const user = await User.findOne({ email });

    if (!user) {
        console.log('User not found!');
        process.exit(1);
    }

    console.log(`User Found: ${user.name} (${user._id})`);

    // Simulate what the frontend ACTUALLY sends
    // apiService.js does: new Date().getTimezoneOffset() * -1
    // For UTC+2, getTimezoneOffset is -120. So it sends 120.
    const tzOffset = 120;
    const rawOffset = Number(tzOffset);

    // Current backend logic (buggy):
    const localOffset = -rawOffset; // Becomes -120 (UTC-2)

    const today = moment().utcOffset(localOffset);
    const startOfDay = today.clone().startOf('day').toDate();
    const endOfDay = today.clone().endOf('day').toDate();

    console.log('Debug Timezone:', {
        tzOffset,
        localOffset,
        serverTime: new Date(),
        startOfDay,
        endOfDay
    });

    // 1. Check Personal Tasks
    const sections = await Sections.find({ userId: user._id });
    console.log(`Found ${sections.length} sections for user.`);

    let personalCount = 0;
    sections.forEach(section => {
        section.tasks.forEach(task => {
            if (task.dueDate && !task.isDone) {
                const taskMoment = moment(task.dueDate).utcOffset(localOffset);

                // Check if it falls in today
                if (taskMoment.isBetween(moment(startOfDay), moment(endOfDay), undefined, '[]')) {
                    console.log(`[MATCH] Personal Task: "${task.name}" | Due: ${task.dueDate} | Local: ${taskMoment.format()}`);
                    personalCount++;
                } else {
                    // Log near misses
                    const diff = Math.abs(taskMoment.diff(moment(startOfDay), 'hours'));
                    if (diff < 48) {
                        console.log(`[MISS] Personal Task: "${task.name}" | Due: ${task.dueDate} | Local: ${taskMoment.format()} | Diff: ${diff}h`);
                    }
                }
            }
        });
    });

    // 2. Check Group Tasks
    const emailRegex = new RegExp(`^${email}$`, 'i');
    const groupTasks = await GroupTask.find({
        dueDate: { $gte: startOfDay, $lte: endOfDay },
        completed: false,
        $or: [
            { assignedTo: { $in: [emailRegex] } },
            { createdBy: user._id }
        ]
    });

    console.log(`Found ${groupTasks.length} group tasks for today.`);
    groupTasks.forEach(t => console.log(`[MATCH] Group Task: "${t.title}"`));

    // 3. Check Events
    const events = await Event.find({
        userId: user._id,
        $and: [
            { startDate: { $lte: endOfDay } },
            { endDate: { $gte: startOfDay } }
        ]
    });
    console.log(`Found ${events.length} events for today.`);
    events.forEach(e => console.log(`[MATCH] Event: "${e.title}"`));

    console.log('--- SUMMARY ---');
    console.log(`Personal: ${personalCount}`);
    console.log(`Group: ${groupTasks.length}`);
    console.log(`Events: ${events.length}`);
    console.log(`Total: ${personalCount + groupTasks.length + events.length}`);

    process.exit(0);
};

debugUser();
