const express = require('express');
const router = express.Router();
const titiController = require('../controllers/titiController');
const { protect } = require('../middleware/authMiddleware');

/**
 * @swagger
 * components:
 *   schemas:
 *     TitiChatRequest:
 *       type: object
 *       required:
 *         - message
 *       properties:
 *         message:
 *           type: string
 *           description: User's message to Titi
 *       example:
 *         message: "أنشالي workspace اسمها Study Plan"
 *     
 *     TitiChatResponse:
 *       type: object
 *       properties:
 *         success:
 *           type: boolean
 *         data:
 *           type: object
 *           properties:
 *             message:
 *               type: string
 *               description: Titi's response message
 *             intent:
 *               type: string
 *               description: Detected intent from user message
 *             actionResult:
 *               type: object
 *               description: Result of executed action if any
 *             timestamp:
 *               type: string
 *               format: date-time
 */

/**
 * @swagger
 * /api/titi/chat:
 *   post:
 *     summary: Send message to Titi chatbot
 *     tags: [Titi]
 *     security:
 *       - bearerAuth: []
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             $ref: '#/components/schemas/TitiChatRequest'
 *     responses:
 *       200:
 *         description: Successful response from Titi
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/TitiChatResponse'
 *       400:
 *         description: Bad request - message is required
 *       401:
 *         description: Unauthorized - authentication required
 *       500:
 *         description: Internal server error
 */
router.post('/chat', protect, titiController.chat);

/**
 * @swagger
 * /api/titi/greeting:
 *   get:
 *     summary: Get personalized greeting from Titi
 *     tags: [Titi]
 *     security:
 *       - bearerAuth: []
 *     responses:
 *       200:
 *         description: Personalized greeting message
 *         content:
 *           application/json:
 *             schema:
 *               type: object
 *               properties:
 *                 success:
 *                   type: boolean
 *                 data:
 *                   type: object
 *                   properties:
 *                     greeting:
 *                       type: string
 *                       description: Personalized greeting message
 *                     userName:
 *                       type: string
 *                       description: User's name
 *                     timestamp:
 *                       type: string
 *                       format: date-time
 *       401:
 *         description: Unauthorized - authentication required
 *       404:
 *         description: User not found
 *       500:
 *         description: Internal server error
 */
router.get('/greeting', protect, titiController.getGreeting);

/**
 * @swagger
 * /api/titi/history:
 *   get:
 *     summary: Get chat history with Titi
 *     tags: [Titi]
 *     security:
 *       - bearerAuth: []
 *     parameters:
 *       - in: query
 *         name: limit
 *         schema:
 *           type: integer
 *           default: 20
 *         description: Number of recent messages to retrieve
 *     responses:
 *       200:
 *         description: Chat history retrieved successfully
 *         content:
 *           application/json:
 *             schema:
 *               type: object
 *               properties:
 *                 success:
 *                   type: boolean
 *                 data:
 *                   type: object
 *                   properties:
 *                     messages:
 *                       type: array
 *                       items:
 *                         type: object
 *                         properties:
 *                           id:
 *                             type: string
 *                           text:
 *                             type: string
 *                           sender:
 *                             type: string
 *                             enum: [user, titi]
 *                           timestamp:
 *                             type: string
 *                             format: date-time
 *                           intent:
 *                             type: string
 *                           actionResult:
 *                             type: object
 *                     count:
 *                       type: integer
 *       401:
 *         description: Unauthorized - authentication required
 *       500:
 *         description: Internal server error
 */
router.get('/history', protect, titiController.getChatHistory);

/**
 * @swagger
 * /api/titi/clear-history:
 *   delete:
 *     summary: Clear chat history with Titi
 *     tags: [Titi]
 *     security:
 *       - bearerAuth: []
 *     responses:
 *       200:
 *         description: Chat history cleared successfully
 *         content:
 *           application/json:
 *             schema:
 *               type: object
 *               properties:
 *                 success:
 *                   type: boolean
 *                 message:
 *                   type: string
 *       401:
 *         description: Unauthorized - authentication required
 *       500:
 *         description: Internal server error
 */
router.delete('/clear-history', protect, titiController.clearChatHistory);

module.exports = router;