const express = require('express');
const { protect } = require('../middleware/authMiddleware');
const {
    validateObjectIdParam,
    validateObjectIdQuery,
    validateObjectIdBody,
} = require('../middleware/validateObjectId');
const {
    addSection,
    getSection,
    updateSection,
    deleteSection,
    addTask,
    updateTask,
    deleteTask,
    markTaskAsDone,
    shareSection,
    getSharedSection,
    assignTask,
    togglePublicView // Added new controller function
} = require('../controllers/sectionControler');

const router = express.Router();

// Public route for accessing shared sections (no auth required)
router.get('/shared/:token', getSharedSection);

// Protect all other routes
router.use(protect);

// Standard section routes
router.post('/', addSection);
router.get('/:userId', [
    validateObjectIdParam('userId'),
    validateObjectIdQuery('workspaceId'),
], getSection);
router.put('/:id', validateObjectIdParam('id'), updateSection);
router.delete('/:id', validateObjectIdParam('id'), deleteSection);

// Task routes
router.post('/:sectionId/tasks', validateObjectIdParam('sectionId'), addTask);
router.put('/:sectionId/tasks/:taskId', [
    validateObjectIdParam('sectionId'),
    validateObjectIdParam('taskId'),
], updateTask);
router.delete('/:sectionId/tasks/:taskId', [
    validateObjectIdParam('sectionId'),
    validateObjectIdParam('taskId'),
], deleteTask);
router.put('/:sectionId/tasks/:taskId/done', [
    validateObjectIdParam('sectionId'),
    validateObjectIdParam('taskId'),
], markTaskAsDone);

// New sharing and assignment routes
router.post('/:sectionId/share', validateObjectIdParam('sectionId'), shareSection);
router.post('/:sectionId/tasks/:taskId/assign', [
    validateObjectIdParam('sectionId'),
    validateObjectIdParam('taskId'),
], assignTask);

// Route to toggle public view status of a section
router.put('/:sectionId/toggle-public-view', validateObjectIdParam('sectionId'), togglePublicView);


module.exports = router;