const express = require('express');
const router = express.Router();
const { protect } = require('../middleware/authMiddleware');
const {
    getCalendarData,
    createEvent,
    updateEvent,
    deleteEvent,
    getEventsForDate,
    getTodayItems
} = require('../controllers/calendarController');

/**
 * @swagger
 * components:
 *   schemas:
 *     Event:
 *       type: object
 *       required:
 *         - title
 *         - startDate
 *       properties:
 *         _id:
 *           type: string
 *           description: The auto-generated id of the event
 *         title:
 *           type: string
 *           description: The event title
 *         description:
 *           type: string
 *           description: The event description
 *         startDate:
 *           type: string
 *           format: date-time
 *           description: The start date and time of the event
 *         endDate:
 *           type: string
 *           format: date-time
 *           description: The end date and time of the event
 *         allDay:
 *           type: boolean
 *           description: Whether the event is all day
 *         userId:
 *           type: string
 *           description: The user who created the event
 *         workspaceId:
 *           type: string
 *           description: The workspace the event belongs to
 *         reminder:
 *           type: object
 *           properties:
 *             enabled:
 *               type: boolean
 *             minutes:
 *               type: number
 *         color:
 *           type: string
 *           description: The color code for the event
 *         type:
 *           type: string
 *           enum: [event, meeting, reminder, other]
 *           description: The type of event
 *       example:
 *         title: "Team Meeting"
 *         description: "Weekly team sync meeting"
 *         startDate: "2024-01-20T14:00:00.000Z"
 *         endDate: "2024-01-20T15:00:00.000Z"
 *         allDay: false
 *         reminder:
 *           enabled: true
 *           minutes: 15
 *         color: "#10B981"
 *         type: "meeting"
 */

/**
 * @swagger
 * tags:
 *   name: Calendar
 *   description: Calendar management API
 */

/**
 * @swagger
 * /api/calendar:
 *   get:
 *     summary: Get all calendar data (events + tasks) for a user
 *     tags: [Calendar]
 *     security:
 *       - bearerAuth: []
 *     parameters:
 *       - in: query
 *         name: workspaceId
 *         schema:
 *           type: string
 *         description: Filter by workspace ID
 *       - in: query
 *         name: startDate
 *         schema:
 *           type: string
 *           format: date
 *         description: Start date for filtering (YYYY-MM-DD)
 *       - in: query
 *         name: endDate
 *         schema:
 *           type: string
 *           format: date
 *         description: End date for filtering (YYYY-MM-DD)
 *     responses:
 *       200:
 *         description: Calendar data retrieved successfully
 *         content:
 *           application/json:
 *             schema:
 *               type: object
 *               properties:
 *                 success:
 *                   type: boolean
 *                 data:
 *                   type: object
 *                   properties:
 *                     items:
 *                       type: array
 *                       items:
 *                         $ref: '#/components/schemas/Event'
 *                     groupedByDate:
 *                       type: object
 *                     summary:
 *                       type: object
 *       401:
 *         description: Unauthorized
 *       500:
 *         description: Server error
 */
router.get('/', protect, getCalendarData);

/**
 * @swagger
 * /api/calendar/events:
 *   post:
 *     summary: Create a new event
 *     tags: [Calendar]
 *     security:
 *       - bearerAuth: []
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             type: object
 *             required:
 *               - title
 *               - startDate
 *             properties:
 *               title:
 *                 type: string
 *               description:
 *                 type: string
 *               startDate:
 *                 type: string
 *                 format: date-time
 *               endDate:
 *                 type: string
 *                 format: date-time
 *               allDay:
 *                 type: boolean
 *               workspaceId:
 *                 type: string
 *               reminder:
 *                 type: object
 *                 properties:
 *                   enabled:
 *                     type: boolean
 *                   minutes:
 *                     type: number
 *               color:
 *                 type: string
 *     responses:
 *       201:
 *         description: Event created successfully
 *         content:
 *           application/json:
 *             schema:
 *               type: object
 *               properties:
 *                 success:
 *                   type: boolean
 *                 message:
 *                   type: string
 *                 data:
 *                   $ref: '#/components/schemas/Event'
 *       400:
 *         description: Bad request
 *       401:
 *         description: Unauthorized
 *       500:
 *         description: Server error
 */
router.post('/events', protect, createEvent);

/**
 * @swagger
 * /api/calendar/events/{eventId}:
 *   put:
 *     summary: Update an event
 *     tags: [Calendar]
 *     security:
 *       - bearerAuth: []
 *     parameters:
 *       - in: path
 *         name: eventId
 *         required: true
 *         schema:
 *           type: string
 *         description: The event ID
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             type: object
 *             properties:
 *               title:
 *                 type: string
 *               description:
 *                 type: string
 *               startDate:
 *                 type: string
 *                 format: date-time
 *               endDate:
 *                 type: string
 *                 format: date-time
 *               allDay:
 *                 type: boolean
 *               reminder:
 *                 type: object
 *               color:
 *                 type: string
 *     responses:
 *       200:
 *         description: Event updated successfully
 *       400:
 *         description: Bad request
 *       401:
 *         description: Unauthorized
 *       404:
 *         description: Event not found
 *       500:
 *         description: Server error
 */
router.put('/events/:eventId', protect, updateEvent);

/**
 * @swagger
 * /api/calendar/events/{eventId}:
 *   delete:
 *     summary: Delete an event
 *     tags: [Calendar]
 *     security:
 *       - bearerAuth: []
 *     parameters:
 *       - in: path
 *         name: eventId
 *         required: true
 *         schema:
 *           type: string
 *         description: The event ID
 *     responses:
 *       200:
 *         description: Event deleted successfully
 *       401:
 *         description: Unauthorized
 *       404:
 *         description: Event not found
 *       500:
 *         description: Server error
 */
router.delete('/events/:eventId', protect, deleteEvent);

/**
 * @swagger
 * /api/calendar/date:
 *   get:
 *     summary: Get events and tasks for a specific date
 *     tags: [Calendar]
 *     security:
 *       - bearerAuth: []
 *     parameters:
 *       - in: query
 *         name: date
 *         required: true
 *         schema:
 *           type: string
 *           format: date
 *         description: The date to get events for (YYYY-MM-DD)
 *       - in: query
 *         name: workspaceId
 *         schema:
 *           type: string
 *         description: Filter by workspace ID
 *     responses:
 *       200:
 *         description: Events and tasks for date retrieved successfully
 *       400:
 *         description: Bad request - date parameter required
 *       401:
 *         description: Unauthorized
 *       500:
 *         description: Server error
 */
router.get('/date', protect, getEventsForDate);

/**
 * @swagger
 * /api/calendar/today:
 *   get:
 *     summary: Get today's tasks and events for notifications
 *     tags: [Calendar]
 *     security:
 *       - bearerAuth: []
 *     parameters:
 *       - in: query
 *         name: workspaceId
 *         schema:
 *           type: string
 *         description: Filter by workspace ID
 *     responses:
 *       200:
 *         description: Today's items retrieved successfully
 *         content:
 *           application/json:
 *             schema:
 *               type: object
 *               properties:
 *                 success:
 *                   type: boolean
 *                 data:
 *                   type: object
 *                   properties:
 *                     events:
 *                       type: array
 *                       items:
 *                         $ref: '#/components/schemas/Event'
 *                     tasks:
 *                       type: array
 *                     summary:
 *                       type: object
 *                       properties:
 *                         totalEvents:
 *                           type: number
 *                         totalTasks:
 *                           type: number
 *                         message:
 *                           type: string
 *       401:
 *         description: Unauthorized
 *       500:
 *         description: Server error
 */
router.get('/today', protect, getTodayItems);

module.exports = router;