const mongoose = require('mongoose');

const titiChatHistorySchema = new mongoose.Schema({
  userId: {
    type: mongoose.Schema.Types.ObjectId,
    ref: 'User',
    required: true,
    index: true
  },
  messages: [{
    id: {
      type: String,
      required: true
    },
    text: {
      type: String,
      required: true
    },
    sender: {
      type: String,
      enum: ['user', 'titi'],
      required: true
    },
    timestamp: {
      type: Date,
      default: Date.now
    },
    intent: {
      type: String,
      default: null // For storing detected intents like 'create_task', 'create_workspace', etc.
    },
    actionResult: {
      type: mongoose.Schema.Types.Mixed,
      default: null // For storing results of executed actions
    }
  }],
  lastActivity: {
    type: Date,
    default: Date.now
  }
}, {
  timestamps: true
});

// Index for automatic cleanup after 2 days
titiChatHistorySchema.index({ "lastActivity": 1 }, { expireAfterSeconds: 172800 }); // 2 days = 172800 seconds

// Update lastActivity on any message addition
titiChatHistorySchema.pre('save', function(next) {
  this.lastActivity = new Date();
  next();
});

// Static method to add message to user's chat history
titiChatHistorySchema.statics.addMessage = async function(userId, message) {
  try {
    let chatHistory = await this.findOne({ userId });
    
    if (!chatHistory) {
      chatHistory = new this({
        userId,
        messages: [message]
      });
    } else {
      chatHistory.messages.push(message);
      // Keep only last 50 messages to prevent excessive storage
      if (chatHistory.messages.length > 50) {
        chatHistory.messages = chatHistory.messages.slice(-50);
      }
    }
    
    await chatHistory.save();
    return chatHistory;
  } catch (error) {
    console.error('Error adding message to chat history:', error);
    throw error;
  }
};

// Static method to get user's recent chat history
titiChatHistorySchema.statics.getRecentHistory = async function(userId, limit = 10) {
  try {
    const chatHistory = await this.findOne({ userId });
    if (!chatHistory) return [];
    
    // Return last 'limit' messages
    return chatHistory.messages.slice(-limit);
  } catch (error) {
    console.error('Error getting chat history:', error);
    throw error;
  }
};

module.exports = mongoose.model('TitiChatHistory', titiChatHistorySchema);