const groqService = require('../utils/groqService');
const TitiChatHistory = require('../models/TitiChatHistory');
const User = require('../models/User');
const Workspace = require('../models/Workspace');
const Sections = require('../models/Sections');
const GroupTask = require('../models/GroupTask');
const GroupSubtask = require('../models/GroupSubtask');

class TitiController {
  // Main chat endpoint
  async chat(req, res) {
    try {
      const { message } = req.body;
      const userId = req.user.id;

      if (!message || !message.trim()) {
        return res.status(400).json({
          success: false,
          message: 'Message is required'
        });
      }

      // Get user info for personalization
      const user = await User.findById(userId);
      if (!user) {
        return res.status(404).json({
          success: false,
          message: 'User not found'
        });
      }

      // Check premium status and AI usage limits for free users
      const now = new Date();
      const isPremiumActive = user.premiumActive && 
                             user.premiumExpiry && 
                             new Date(user.premiumExpiry) > now;
      
      if (!isPremiumActive) {
        // Reset daily counter if it's a new day
        const lastReset = user.aiUsage?.lastResetDate || new Date();
        const isNewDay = lastReset.toDateString() !== now.toDateString();
        
        if (isNewDay) {
          user.aiUsage = {
            dailyQueries: 0,
            lastResetDate: now
          };
          await user.save();
        }

        // Check daily limit (10 queries per day for free users)
        const dailyQueries = user.aiUsage?.dailyQueries || 0;
        if (dailyQueries >= 10) {
          return res.status(403).json({
            success: false,
            message: 'Sorry, you have used all 10 daily Titi chat requests. Subscribe to Premium to enjoy unlimited Titi chat conversations!',
            code: 'AI_LIMIT_REACHED',
            dailyQueries: dailyQueries,
            maxQueries: 10
          });
        }

        // Increment query count for free users
        user.aiUsage = {
          dailyQueries: dailyQueries + 1,
          lastResetDate: isNewDay ? now : (user.aiUsage?.lastResetDate || now)
        };
        await user.save();
      }

      // Get user's task data for context
      const taskContext = await this.getUserTaskContext(userId);

      // Get recent chat history for context
      const chatHistory = await TitiChatHistory.getRecentHistory(userId, 5);

      // If message contains clear plan/action/entity cues, skip small-talk handling entirely
      const hasPlanCue = /(خطة|plan|برنامج|جدول|هيكل|مذاكرة|study|مراجعة|مذاكره)/i.test(message || '');
      const hasActionVerb = /(اعمل|اعملي|أنشئ|انشئ|إنشاء|انشاء|ضيف|أضف|حط|رتب|نظم|جهز|create|make|add|build|اريد|أريد|عايز|عاوز|ارغب|أرغب|عايزة|عاوزة)/i.test(message || '');
      const hasEntityCue = /(workspace|وورك\s*سبيس|مساحة\s*عمل|section|سكشن|task|تاسك|مهمة|قسم|ورك\s*سبيس|ووركسبيس)/i.test(message || '');

      // Lightweight small-talk/greeting/name handling to avoid misunderstandings
      const smallTalk = (!hasPlanCue && !hasActionVerb && !hasEntityCue) ? this.detectSmallTalk(message, user) : null;
      if (smallTalk) {
        const userMessage = {
          id: Date.now().toString(),
          text: message,
          sender: 'user',
          timestamp: new Date()
        };
        await TitiChatHistory.addMessage(userId, userMessage);

        const titiMessage = {
          id: (Date.now() + 1).toString(),
          text: smallTalk.response,
          sender: 'titi',
          timestamp: new Date(),
          intent: smallTalk.intent || 'general_chat',
          actionResult: null
        };
        await TitiChatHistory.addMessage(userId, titiMessage);

        return res.json({
          success: true,
          data: {
            message: smallTalk.response,
            intent: smallTalk.intent || 'general_chat',
            actionResult: null,
            actions: null,
            timestamp: new Date()
          }
        });
      }

      // Check if we are waiting for a due date from the user
      const lastTitiMsg = Array.isArray(chatHistory) ? [...chatHistory].reverse().find(m => m.sender === 'titi') : null;
      const pendingDue = lastTitiMsg && lastTitiMsg.actionResult && lastTitiMsg.actionResult.pendingAction === 'create_task_due_date'
        ? lastTitiMsg.actionResult
        : null;

      let aiResult = null;
      let interceptedActionResult = null;
      let interceptedResponseText = null;

      if (pendingDue && pendingDue.pendingParams) {
        // Try to parse due date from the user's message
        const parsedDue = this.parseDueDateText(message);
        const askCount = Number(pendingDue.askCount || 1);
        if (parsedDue) {
          // Create task now with parsed due date
          interceptedActionResult = await this.createTask({ ...pendingDue.pendingParams, dueDate: parsedDue }, userId);
          const humanDate = new Date(parsedDue).toLocaleDateString();
          interceptedResponseText = `✅ تم إنشاء المهمة بتاريخ الاستحقاق: ${humanDate}.`;
        } else if (askCount >= 2) {
          // Default to today if user still doesn't provide a due date
          const today = new Date();
          interceptedActionResult = await this.createTask({ ...pendingDue.pendingParams, dueDate: today }, userId);
          const humanDate = today.toLocaleDateString();
          interceptedResponseText = `⏰ لم يتم تقديم موعد الاستحقاق، قمت بتعيين اليوم (${humanDate}).`;
        } else {
          // Ask again for due date with clearer instruction
          interceptedActionResult = {
            success: false,
            action: 'ask_due_date',
            data: { pendingParams: pendingDue.pendingParams, askCount: askCount + 1 }
          };
          interceptedResponseText = 'ما تاريخ الاستحقاق لهذه المهمة؟ اكتب تاريخًا مثل 2025-11-08، أو "اليوم"/"بكرة".';
        }
      } else {
        // Generate AI response with task context
        aiResult = await groqService.generateResponse(
          message,
          chatHistory,
          { userName: user.name, userId, taskContext }
        );

        // Safety guard: prevent unintended study plan creation unless explicitly requested,
        // BUT respect ongoing plan context from recent history.
        if (aiResult) {
          const userAskedPlanNow = /(خطة|plan|برنامج|جدول|هيكل)/i.test(message);
          const isMulti = Array.isArray(aiResult.actions) && aiResult.actions.length >= 3;
          const hasDBPlanName = Array.isArray(aiResult.actions) && aiResult.actions.some(a => a?.intent === 'create_workspace' && /Study\s*Plan|DB/i.test((a?.params?.name || '')));
          const planIntent = aiResult.intent === 'get_plan';
          const inPlanContext = this.isPlanContext(chatHistory);
          if (!userAskedPlanNow && !inPlanContext && (planIntent || isMulti || hasDBPlanName)) {
            aiResult = {
              response: 'لو محتاج خطة منظمة، قول لي صراحة: "عايز خطة" أو "خطط للدراسة" 👍',
              intent: 'general_chat',
              params: {},
              actions: null
            };
          }
        }
      }

      // Save user message to history
      const userMessage = {
        id: Date.now().toString(),
        text: message,
        sender: 'user',
        timestamp: new Date()
      };
      await TitiChatHistory.addMessage(userId, userMessage);

      // Process single intent or multi-actions if detected
      let actionResult = null;
      let actionResults = null;
      if (!interceptedActionResult && aiResult && Array.isArray(aiResult.actions) && aiResult.actions.length > 0) {
        actionResults = await this.processActionsPipeline(aiResult.actions, userId);
      } else if (!interceptedActionResult && aiResult && aiResult.intent) {
        actionResult = await this.processIntent(aiResult.intent, aiResult.params, userId);
      }

      // If create_task returned ask_due_date, override the displayed message and set pending state
      let finalMessageText = aiResult?.response || '';
      let finalIntent = Array.isArray(aiResult?.actions) ? 'multi' : aiResult?.intent || null;
      let finalActionResult = Array.isArray(aiResult?.actions) ? actionResults : actionResult;

      if (interceptedActionResult) {
        finalActionResult = interceptedActionResult;
        finalIntent = 'create_task';
        finalMessageText = interceptedResponseText;
      } else if (finalIntent === 'create_task' && finalActionResult && finalActionResult.action === 'ask_due_date') {
        finalMessageText = 'ما تاريخ الاستحقاق لهذه المهمة؟ اكتب تاريخًا مثل 2025-11-08، أو "اليوم"/"بكرة".';
      }

      // Save Titi's response to history
      const titiMessage = {
        id: (Date.now() + 1).toString(),
        text: finalMessageText,
        sender: 'titi',
        timestamp: new Date(),
        intent: finalIntent,
        actionResult: (() => {
          // If we need to remember a pending due date request, store a marker
          if ((finalActionResult && finalActionResult.action === 'ask_due_date') || interceptedResponseText === 'ما تاريخ الاستحقاق لهذه المهمة؟ اكتب تاريخًا مثل 2025-11-08، أو "اليوم"/"بكرة".') {
            const pendingParams = finalActionResult?.data?.pendingParams || (pendingDue ? pendingDue.pendingParams : null);
            const askCount = finalActionResult?.data?.askCount || ((pendingDue?.askCount || 1) + 1);
            return { pendingAction: 'create_task_due_date', pendingParams, askCount };
          }
          return finalActionResult;
        })()
      };
      await TitiChatHistory.addMessage(userId, titiMessage);

      res.json({
        success: true,
        data: {
          message: finalMessageText,
          intent: finalIntent,
          actionResult: finalActionResult,
          actions: Array.isArray(aiResult?.actions) ? aiResult.actions : null,
          timestamp: new Date()
        }
      });

    } catch (error) {
      console.error('Titi chat error:', error);
      res.status(500).json({
        success: false,
        message: 'حدث خطأ تقني، سأعود قريباً! 🔧',
        error: error.message
      });
    }
  }

  // Detect simple small-talk intents to respond without invoking the planner
  detectSmallTalk(text = '', user = {}) {
    // Normalize Arabic text: remove diacritics, tatweel, normalize alef/yeh variations
    const raw = String(text || '').trim();
    const stripDiacritics = (s) => s.replace(/[\u064B-\u065F]/g, ''); // Arabic diacritics range
    const stripTatweel = (s) => s.replace(/[\u0640]/g, ''); // Tatweel
    const normalizeLetters = (s) => s
      .replace(/[\u0622\u0623\u0625]/g, '\u0627') // normalize alef variants to alef
      .replace(/[\u0649]/g, '\u064A'); // normalize alef maqsura to yeh
    const prepped = normalizeLetters(stripTatweel(stripDiacritics(raw))).toLowerCase();
    const msg = prepped;
    const normalized = msg.replace(/\s+/g, ' ').trim();
    const arabic = (s) => new RegExp(s, 'i');

    // If the message contains clear action/plan/entity cues, do NOT treat it as small talk
    const actionVerbs = /(اعمل|اعملي|اعمالي|أنشئ|انشئ|إنشاء|انشاء|ضيف|أضف|حط|رتب|نظم|جهز|create|make|add|build|اريد|أريد|عايز|عاوز|ارغب|أرغب|عايزة|عاوزة)/i;
    const planCues = /(خطة|plan|برنامج|جدول|هيكل|مذاكرة|study|مراجعة|مذاكره)/i;
    const entityCues = /(workspace|وورك\s*سبيس|مساحة\s*عمل|section|سكشن|task|تاسك|مهمة|قسم|ورك\s*سبيس|ووركسبيس)/i;
    const hasAction = actionVerbs.test(normalized) || planCues.test(normalized) || entityCues.test(normalized);
    if (hasAction) return null;

    // Name questions
    const askName = /(اسمك\s*ايه|اسمك\s*ايه\?|انت\s*اسمك\s*ايه|اسمك\?|ما\s*اسمك|مين\s*انت)/i;
    if (askName.test(msg)) {
      const userNamePart = user?.name ? `يا ${user.name}` : '';
      return {
        intent: 'small_talk_name',
        response: `أنا اسمي تيتي 👻، مساعدك الذكي في TaskTree ${userNamePart}. جاهزة أساعدك في أي وقت!`
      };
    }

    // Greetings / well-being
    const greetings = /(ازيك|إزيك|ازيكم|عامل\s*ايه|عاملة\s*ايه|ايه\s*الاخبار|كيف\s*الحال|كيفك|كيف\s*حالك|مرحبا|مرحباً|اهلا|أهلاً|هاي|هلا|سلام|صباح\s*الخير|مساء\s*الخير)/i;
    const wellbeing = /(صحتك\s*عاملة\s*ايه|مالك|أخبارك|اخبارك\s*ايه|كويس|بخير|كيف\s*حالك)/i;
    if (!hasAction && (greetings.test(msg) || wellbeing.test(msg))) {
      return {
        intent: 'small_talk_greeting',
        response: 'أنا تمام والحمد لله! 😊 جاهزة أساعدك في أي حاجة. تحب نبدأ بحاجة معينة؟'
      };
    }

    // Acknowledgements / closing
    const thanksOrClose = /(شكرا|شكراً|ثانكس|thanks|خلاص)/i;
    const isJustOkay = /^تمام(?:\s+شكرا)?!?$/i.test(normalized);
    if (!hasAction && (thanksOrClose.test(normalized) || isJustOkay)) {
      return {
        intent: 'small_talk_ack',
        response: 'العفو! لو احتجت أي حاجة، أنا هنا دايمًا 😊'
      };
    }

    return null;
  }

  // Detect if user is already in a planning context in recent history
  isPlanContext(chatHistory = []) {
    try {
      if (!Array.isArray(chatHistory) || chatHistory.length === 0) return false;
      // Look at recent 5 messages: if user or Titi mentioned plan cues or last Titi had plan/multi actions
      const recent = chatHistory.slice(-5);
      const planCue = /(خطة|plan|برنامج|جدول|هيكل|سكاشن|sections)/i;
      for (const msg of recent) {
        if (planCue.test(msg.text || '')) return true;
        if (msg.sender === 'titi' && (msg.intent === 'get_plan' || msg.intent === 'multi')) return true;
        const ar = msg.actionResult;
        if (ar) {
          // If actionResult indicates workspace/section/task creation in sequence, assume planning
          const asText = JSON.stringify(ar);
          if (/workspace_created|section_created|create_task/.test(asText)) return true;
        }
      }
      return false;
    } catch {
      return false;
    }
  }

  // Get user task context for AI responses
  async getUserTaskContext(userId) {
    try {
      const today = new Date();
      today.setHours(0, 0, 0, 0);
      const tomorrow = new Date(today);
      tomorrow.setDate(tomorrow.getDate() + 1);

      // Get user's workspaces with their sections and tasks
      const workspaces = await Workspace.find({ userId })
        .populate({
          path: 'sections',
          select: 'name tasks',
          populate: {
            path: 'tasks',
            select: 'name description isDone priority isImportant dueDate tags'
          }
        })
        .select('name sections');
      
      console.log('DEBUG - getUserTaskContext for userId:', userId);
      console.log('DEBUG - Workspaces found:', workspaces.length);
      workspaces.forEach(workspace => {
        console.log('DEBUG - Workspace:', workspace.name, 'Sections:', workspace.sections?.map(s => s.name) || []);
        workspace.sections?.forEach(section => {
          console.log('DEBUG - Section:', section.name, 'Tasks:', section.tasks?.length || 0);
        });
      });

      // Get urgent tasks (high priority or overdue)
      const urgentTasks = await GroupTask.find({
        $or: [
          { userId, priority: 'high', status: { $ne: 'completed' } },
          { userId, dueDate: { $lt: today }, status: { $ne: 'completed' } }
        ]
      })
      .sort({ dueDate: 1, priority: -1 })
      .limit(5)
      .select('name dueDate priority status');

      // Get today's tasks
      const todayTasks = await GroupTask.find({
        userId,
        dueDate: { $gte: today, $lt: tomorrow },
        status: { $ne: 'completed' }
      })
      .sort({ priority: -1 })
      .limit(3)
      .select('name dueDate priority status');

      // Get overdue count
      const overdueCount = await GroupTask.countDocuments({
        userId,
        dueDate: { $lt: today },
        status: { $ne: 'completed' }
      });

      // Get high priority count
      const highPriorityCount = await GroupTask.countDocuments({
        userId,
        priority: 'high',
        status: { $ne: 'completed' }
      });

      return {
        workspaces: workspaces.map(workspace => ({
          name: workspace.name,
          sections: workspace.sections.map(section => ({
            name: section.name,
            tasks: section.tasks.map(task => ({
              name: task.name,
              description: task.description,
              isDone: task.isDone,
              priority: task.priority,
              isImportant: task.isImportant,
              dueDate: task.dueDate,
              tags: task.tags
            }))
          }))
        })),
        urgentTasks: urgentTasks.map(task => ({
          name: task.name,
          dueDate: task.dueDate,
          priority: task.priority,
          isOverdue: task.dueDate < today
        })),
        todayTasks: todayTasks.map(task => ({
          name: task.name,
          dueDate: task.dueDate,
          priority: task.priority
        })),
        overdueCount,
        highPriorityCount
      };
    } catch (error) {
      console.error('Error getting user task context:', error);
      return {
        workspaces: [],
        urgentTasks: [],
        todayTasks: [],
        overdueCount: 0,
        highPriorityCount: 0
      };
    }
  }

  // Process detected intents and execute actions
  async processIntent(intent, params, userId) {
    try {
      switch (intent) {
        case 'create_workspace':
          return await this.createWorkspace(params, userId);
        
        case 'create_section':
          return await this.createSection(params, userId);
        
        case 'create_task':
          return await this.createTask(params, userId);
        
        case 'update_task':
          return await this.updateTask(params, userId);
        
        case 'get_summary':
          return await this.getDailySummary(userId);
        
        case 'get_plan':
          return await this.getSmartPlan(userId);
        
        default:
          return null;
      }
    } catch (error) {
      console.error(`Error processing intent ${intent}:`, error);
      return { error: error.message };
    }
  }

  // Determine if an action can be executed now based on dependencies and DB state
  async canExecuteAction(action, executedIndices, userId) {
    // If explicit dependsOn present, ensure all dependencies are executed
    if (Array.isArray(action.dependsOn)) {
      const allDepsDone = action.dependsOn.every(depIdx => executedIndices.has(depIdx));
      if (!allDepsDone) return false;
    } else if (typeof action.dependsOn === 'number') {
      // Support single numeric dependency
      if (!executedIndices.has(action.dependsOn)) return false;
    }

    const { intent, params = {} } = action;
    try {
      switch (intent) {
        case 'create_workspace':
          return true; // always safe
        case 'create_section': {
          if (params.workspaceId) return true;
          if (params.workspaceName) {
            const ws = await Workspace.findOne({
              name: { $regex: params.workspaceName, $options: 'i' },
              userId
            });
            return !!ws;
          }
          return false; // need a workspace
        }
        case 'create_task': {
          if (params.sectionId) return true;
          // Try lookup by sectionName and optional workspaceName
          if (params.sectionName) {
            let query = {
              name: { $regex: params.sectionName, $options: 'i' },
              userId
            };
            if (params.workspaceName) {
              const ws = await Workspace.findOne({
                name: { $regex: params.workspaceName, $options: 'i' },
                userId
              });
              if (ws) query.workspace = ws._id; else return false;
            }
            const sec = await Sections.findOne(query);
            return !!sec;
          }
          return false; // need a section
        }
        case 'update_task': {
          if (params.taskId) {
            const t = await GroupTask.findOne({ _id: params.taskId, userId });
            return !!t;
          }
          if (params.taskName) {
            const t = await GroupTask.findOne({
              name: { $regex: params.taskName, $options: 'i' },
              userId
            });
            return !!t;
          }
          return false;
        }
        case 'get_summary':
        case 'get_plan':
          return true;
        default:
          return false;
      }
    } catch (e) {
      return false;
    }
  }

  // Execute a single action via the intent processor
  async executeAction(action, userId) {
    try {
      const result = await this.processIntent(action.intent, action.params || {}, userId);
      return { status: 'fulfilled', intent: action.intent, params: action.params || {}, result };
    } catch (error) {
      return { status: 'rejected', intent: action.intent, params: action.params || {}, error: error.message };
    }
  }

  // Process a list of actions with dependency-aware, concurrent execution where possible
  async processActionsPipeline(actions, userId) {
    const pending = actions.map((a, idx) => ({ ...a, index: idx }));
    const executed = new Set();
    const results = [];

    // Iterate until all actions are executed or no progress is possible
    const maxIterations = actions.length * 5;
    let iterations = 0;

    while (pending.length > 0 && iterations < maxIterations) {
      iterations++;

      // Determine which actions are ready to run now
      const readinessChecks = await Promise.all(
        pending.map(a => this.canExecuteAction(a, executed, userId))
      );

      const ready = pending.filter((a, i) => readinessChecks[i]);

      if (ready.length === 0) {
        // Fallback: try to execute the first pending action that is safest (e.g., create_workspace)
        const safeIdx = pending.findIndex(a => a.intent === 'create_workspace');
        const pick = safeIdx >= 0 ? pending[safeIdx] : pending[0];

        const execRes = await this.executeAction(pick, userId);
        results.push({ index: pick.index, ...execRes });
        executed.add(pick.index);
        pending.splice(pending.findIndex(a => a.index === pick.index), 1);
        continue;
      }

      // Execute all ready actions concurrently
      const execPromises = ready.map(a => this.executeAction(a, userId));
      const settled = await Promise.all(execPromises);

      // Record results and remove from pending
      for (let i = 0; i < ready.length; i++) {
        const a = ready[i];
        const res = settled[i];
        results.push({ index: a.index, ...res });
        executed.add(a.index);
      }

      // Remove executed actions from pending
      for (const a of ready) {
        const idx = pending.findIndex(p => p.index === a.index);
        if (idx >= 0) pending.splice(idx, 1);
      }
    }

    // If some actions remain unexecuted, mark them as blocked
    for (const a of pending) {
      results.push({ index: a.index, status: 'blocked', intent: a.intent, params: a.params || {}, error: 'Dependencies not satisfied' });
    }

    // Sort results by original index order
    results.sort((r1, r2) => r1.index - r2.index);
    return results;
  }

  // Create workspace
  async createWorkspace(params, userId) {
    try {
      const { name, description } = params;
      
      if (!name) {
        throw new Error('Workspace name is required');
      }

      // Idempotency: return existing workspace if same name already exists for user
      const existingWorkspace = await Workspace.findOne({
        userId,
        name: { $regex: `^${name}$`, $options: 'i' }
      });

      if (existingWorkspace) {
        return {
          success: true,
          action: 'workspace_exists',
          data: existingWorkspace
        };
      }

      const workspace = new Workspace({
        name,
        description: description || '',
        userId,
        isDefault: false
      });

      await workspace.save();

      return {
        success: true,
        action: 'workspace_created',
        data: workspace
      };
    } catch (error) {
      throw error;
    }
  }

  // Create section
  async createSection(params, userId) {
    try {
      const { name, workspaceId, workspaceName } = params;
      
      if (!name) {
        throw new Error('Section name is required');
      }

      let targetWorkspaceId = workspaceId;

      // If workspace name is provided, find the workspace
      if (workspaceName && !workspaceId) {
        const workspace = await Workspace.findOne({ 
          name: { $regex: workspaceName, $options: 'i' }, 
          userId 
        });
        if (workspace) {
          targetWorkspaceId = workspace._id;
        } else {
          throw new Error(`Workspace "${workspaceName}" not found`);
        }
      }

      // If no workspace specified, use default workspace
      if (!targetWorkspaceId) {
        const defaultWorkspace = await Workspace.findOne({ userId, isDefault: true });
        if (defaultWorkspace) {
          targetWorkspaceId = defaultWorkspace._id;
        } else {
          throw new Error('No workspace found');
        }
      }

      // Idempotency: return existing section if same name already exists in workspace for user
      const existingSection = await Sections.findOne({
        userId,
        workspace: targetWorkspaceId,
        name: { $regex: `^${name}$`, $options: 'i' }
      });

      if (existingSection) {
        const updatedWorkspace = await Workspace.findById(targetWorkspaceId).populate('sections');
        return {
          success: true,
          action: 'section_exists',
          data: {
            section: existingSection,
            workspace: updatedWorkspace
          }
        };
      }

      const section = new Sections({
        name,
        userId,
        workspace: targetWorkspaceId
      });

      await section.save();

      // Add section to workspace's sections array
      await Workspace.findByIdAndUpdate(
        targetWorkspaceId,
        { $addToSet: { sections: section._id } }
      );

      // Get the updated workspace with populated sections
      const updatedWorkspace = await Workspace.findById(targetWorkspaceId).populate('sections');

      return {
        success: true,
        action: 'section_created',
        data: {
          section: section,
          workspace: updatedWorkspace
        }
      };
    } catch (error) {
      throw error;
    }
  }

  // Create task
  async createTask(params, userId) {
    try {
      const { name, sectionId, sectionName, workspaceName, priority, dueDate, subtasks, description } = params;
      
      if (!name) {
        throw new Error('Task name is required');
      }

      let targetSectionId = sectionId;

      // If section name is provided, find the section
      if (sectionName && !sectionId) {
        let sectionQuery = { 
          name: { $regex: sectionName, $options: 'i' }, 
          userId 
        };

        // If workspace name is also provided, find section within that workspace
        if (workspaceName) {
          const workspace = await Workspace.findOne({
            name: { $regex: workspaceName, $options: 'i' },
            userId
          });
          
          if (workspace) {
            sectionQuery.workspace = workspace._id;
          }
        }

        const section = await Sections.findOne(sectionQuery);
        if (section) {
          targetSectionId = section._id;
        } else {
          // If not found with workspace, try to find any section with that name
          const fallbackSection = await Sections.findOne({ 
            name: { $regex: sectionName, $options: 'i' }, 
            userId 
          });
          if (fallbackSection) {
            targetSectionId = fallbackSection._id;
          } else {
            throw new Error(`Section "${sectionName}" not found`);
          }
        }
      }

      if (!targetSectionId) {
        throw new Error('Section is required to create a task');
      }

      // Fetch section to check for existing task with same name (idempotency)
      const sectionDoc = await Sections.findById(targetSectionId);
      if (!sectionDoc) {
        throw new Error('Section not found');
      }

      const existingTask = sectionDoc.tasks?.find(t => (t.name || '').trim().toLowerCase() === name.trim().toLowerCase());
      if (existingTask) {
        return {
          success: true,
          action: 'task_exists',
          data: {
            task: {
              id: existingTask._id,
              name: existingTask.name,
              priority: existingTask.priority,
              dueDate: existingTask.dueDate,
              sectionId: targetSectionId,
              subtasks: existingTask.subTasks || []
            },
            section: sectionDoc
          }
        };
      }

      // If due date is missing, ask the user for it first
      if (!dueDate) {
        return {
          success: false,
          action: 'ask_due_date',
          data: {
            pendingParams: { name, sectionId: targetSectionId, sectionName, workspaceName, priority, subtasks, description },
            askCount: 1
          }
        };
      }

      // Create the task object
      const newTask = {
        name,
        description: description || '',
        priority: priority || 'medium',
        isImportant: false,
        isDone: false,
        dueDate: new Date(dueDate),
        tags: [],
        subTasks: [],
        assignedTo: [],
        createdBy: userId
      };

      // Add subtasks if provided
      if (subtasks && Array.isArray(subtasks)) {
        newTask.subTasks = subtasks.map(subtaskName => ({
          name: subtaskName,
          isDone: false,
          status: 'pending',
          createdBy: userId,
          description: '',
          priority: 'Medium'
        }));
      }

      // Add the task to the section
      const section = await Sections.findByIdAndUpdate(
        targetSectionId,
        { $push: { tasks: newTask } },
        { new: true }
      );

      if (!section) {
        throw new Error('Section not found');
      }

      // Get the newly created task (last one in the array)
      const createdTask = section.tasks[section.tasks.length - 1];

      return {
        success: true,
        action: 'task_created',
        data: {
          task: {
            id: createdTask._id,
            name: createdTask.name,
            priority: createdTask.priority,
            dueDate: createdTask.dueDate,
            sectionId: targetSectionId,
            subtasks: createdTask.subTasks || []
          },
          section: section
        }
      };
    } catch (error) {
      throw error;
    }
  }

  // Update task
  async updateTask(params, userId) {
    try {
      const { taskId, taskName, status, priority, dueDate } = params;
      
      let task;
      
      if (taskId) {
        task = await GroupTask.findOne({ _id: taskId, userId });
      } else if (taskName) {
        task = await GroupTask.findOne({ 
          name: { $regex: taskName, $options: 'i' }, 
          userId 
        });
      }

      if (!task) {
        throw new Error('Task not found');
      }

      // Update fields if provided
      if (status) task.status = status;
      if (priority) task.priority = priority;
      if (dueDate) task.dueDate = new Date(dueDate);

      await task.save();

      return {
        success: true,
        action: 'task_updated',
        data: {
          id: task._id,
          name: task.name,
          status: task.status,
          priority: task.priority,
          dueDate: task.dueDate
        }
      };
    } catch (error) {
      throw error;
    }
  }

  // Get daily summary
  async getDailySummary(userId) {
    try {
      const today = new Date();
      today.setHours(0, 0, 0, 0);
      const tomorrow = new Date(today);
      tomorrow.setDate(tomorrow.getDate() + 1);

      // Get today's tasks
      const todayTasks = await GroupTask.countDocuments({
        userId,
        dueDate: { $gte: today, $lt: tomorrow }
      });

      // Get overdue tasks
      const overdueTasks = await GroupTask.countDocuments({
        userId,
        dueDate: { $lt: today },
        status: { $ne: 'completed' }
      });

      // Get completed tasks (last 7 days)
      const weekAgo = new Date(today);
      weekAgo.setDate(weekAgo.getDate() - 7);
      const completedTasks = await GroupTask.countDocuments({
        userId,
        status: 'completed',
        updatedAt: { $gte: weekAgo }
      });

      // Get high priority tasks
      const highPriorityTasks = await GroupTask.countDocuments({
        userId,
        priority: 'high',
        status: { $ne: 'completed' }
      });

      const summaryData = {
        todayTasks,
        overdueTasks,
        completedTasks,
        highPriorityTasks
      };

      // Generate AI summary
      const summaryPrompt = groqService.generateSummaryPrompt(summaryData);
      const aiResult = await groqService.generateResponse(summaryPrompt);

      return {
        success: true,
        action: 'daily_summary',
        data: {
          summary: aiResult.response,
          stats: summaryData
        }
      };
    } catch (error) {
      throw error;
    }
  }

  // Get smart plan
  async getSmartPlan(userId) {
    try {
      const today = new Date();
      today.setHours(0, 0, 0, 0);

      // Get pending tasks ordered by priority and due date
      const tasks = await GroupTask.find({
        userId,
        status: { $ne: 'completed' }
      })
      .sort({ priority: -1, dueDate: 1 })
      .limit(10)
      .populate('sectionId', 'name');

      const tasksData = tasks.map(task => ({
        name: task.name,
        priority: task.priority,
        dueDate: task.dueDate,
        section: task.sectionId?.name || 'Unknown'
      }));

      // Generate AI plan
      const planPrompt = groqService.generatePlanPrompt(tasksData);
      const aiResult = await groqService.generateResponse(planPrompt);

      return {
        success: true,
        action: 'smart_plan',
        data: {
          plan: aiResult.response,
          tasks: tasksData
        }
      };
    } catch (error) {
      throw error;
    }
  }

  // Get greeting message
  async getGreeting(req, res) {
    try {
      const userId = req.user.id;
      const userEmail = req.user.email;
      const user = await User.findById(userId);
      
      if (!user) {
        return res.status(404).json({
          success: false,
          message: 'User not found'
        });
      }

      // Get comprehensive task analysis
      const today = new Date();
      today.setHours(0, 0, 0, 0);
      const tomorrow = new Date(today);
      tomorrow.setDate(tomorrow.getDate() + 1);

      // 1. Get personal tasks from sections
      const sections = await Sections.find({ userId }).populate('workspace', 'name');
      let personalTasks = [];
      sections.forEach(section => {
        section.tasks.forEach(task => {
          personalTasks.push({
            name: task.name,
            isDone: task.isDone,
            priority: task.priority,
            dueDate: task.dueDate,
            isImportant: task.isImportant,
            sectionName: section.name,
            workspaceName: section.workspace?.name || 'Personal',
            type: 'personal'
          });
        });
      });

      // 2. Get group tasks (assigned to user or created by user)
      const GroupMembership = require('../models/GroupMembership');
      const Group = require('../models/Group');
      
      const memberships = await GroupMembership.find({ 
        $or: [{ user: userId }, { email: userEmail }] 
      }).populate('group');
      
      let groupTasks = [];
      for (const membership of memberships) {
        const tasks = await GroupTask.find({
          group: membership.group._id,
          $or: [
            { assignedTo: userEmail },
            { createdBy: userId }
          ]
        }).populate('group', 'name');

        tasks.forEach(task => {
          groupTasks.push({
            name: task.title,
            isDone: task.completed,
            priority: task.priority,
            dueDate: task.dueDate,
            points: task.points,
            groupName: task.group.name,
            type: 'group'
          });
        });
      }

      // 3. Combine all tasks for analysis
      const allTasks = [...personalTasks, ...groupTasks];
      
      // 4. Calculate statistics
      const totalTasks = allTasks.length;
      const completedTasks = allTasks.filter(task => task.isDone).length;
      const pendingTasks = totalTasks - completedTasks;
      
      // Urgent tasks (due today or overdue)
      const urgentTasks = allTasks.filter(task => {
        if (!task.dueDate) return false;
        const dueDate = new Date(task.dueDate);
        return dueDate <= tomorrow && !task.isDone;
      });
      
      // Overdue tasks
      const overdueTasks = allTasks.filter(task => {
        if (!task.dueDate) return false;
        const dueDate = new Date(task.dueDate);
        return dueDate < today && !task.isDone;
      });
      
      // High priority pending tasks
      const highPriorityTasks = allTasks.filter(task => 
        (task.priority === 'high' || task.priority === 'High') && !task.isDone
      );
      
      // Recently completed tasks (last 3 days)
      const threeDaysAgo = new Date(today);
      threeDaysAgo.setDate(threeDaysAgo.getDate() - 3);
      const recentlyCompleted = allTasks.filter(task => {
        if (!task.isDone) return false;
        // For personal tasks, check if they were completed recently (we don't have completion date, so we'll assume recent)
        return task.type === 'personal' || task.type === 'group';
      }).slice(0, 3); // Limit to 3 most recent

      // 5. Generate intelligent greeting with analysis
      const timeOfDay = this.getTimeOfDay();
      const taskSummary = {
        totalTasks,
        completedTasks,
        pendingTasks,
        urgentCount: urgentTasks.length,
        overdueCount: overdueTasks.length,
        highPriorityCount: highPriorityTasks.length,
        recentlyCompletedCount: recentlyCompleted.length,
        personalTasksCount: personalTasks.length,
        groupTasksCount: groupTasks.length
      };

      // Generate smart suggestions
      let suggestions = [];
      if (overdueTasks.length > 0) {
        suggestions.push(`لديك ${overdueTasks.length} مهمة متأخرة تحتاج انتباه فوري`);
      }
      if (urgentTasks.length > 0) {
        suggestions.push(`${urgentTasks.length} مهام مستعجلة لهذا اليوم`);
      }
      if (highPriorityTasks.length > 0) {
        suggestions.push(`${highPriorityTasks.length} مهام عالية الأولوية في انتظارك`);
      }
      if (suggestions.length === 0 && pendingTasks > 0) {
        suggestions.push('يمكنك البدء بالمهام ذات الأولوية العالية أولاً');
      }

      // Create welcoming greeting message in English
      let greetingMessage = `👋 ${timeOfDay}, ${user.name}!\n`;
      greetingMessage += `Here's your daily summary for today 📋\n\n`;
      
      greetingMessage += `📅 Today's Tasks\n\n`;
      greetingMessage += `✅ Completed: ${completedTasks}\n\n`;
      greetingMessage += `🔄 Total Tasks: ${totalTasks}\n\n`;
      greetingMessage += `👤 Personal Tasks: ${personalTasks.length}\n\n`;
      greetingMessage += `👥 Group Tasks: ${groupTasks.length}\n\n`;
      
      greetingMessage += `⏰ Status Overview\n\n`;
      
      // Calculate in progress tasks (pending - overdue - blocked)
      const inProgressTasks = pendingTasks - overdueTasks.length;
      if (inProgressTasks > 0) {
        greetingMessage += `🟠 In Progress: ${inProgressTasks}\n\n`;
      }
      
      if (overdueTasks.length > 0) {
        greetingMessage += `⚠️ Delayed: ${overdueTasks.length}\n\n`;
      }
      
      // Add blocked tasks (we'll assume some tasks might be blocked)
      const blockedTasks = Math.floor(pendingTasks * 0.1); // Assume 10% might be blocked
      if (blockedTasks > 0) {
        greetingMessage += `🔒 Blocked: ${blockedTasks}\n\n`;
      }
      
      greetingMessage += `💡 Smart Suggestions\n\n`;
      
      if (overdueTasks.length > 0) {
        greetingMessage += `You have ${overdueTasks.length} delayed tasks, let's handle the most urgent ones first!\n\n`;
      }
      
      if (highPriorityTasks.length > 0) {
        greetingMessage += `🎯 ${highPriorityTasks.length} high-priority tasks need your focus today.\n\n`;
      }
      
      if (completedTasks > 0) {
        greetingMessage += `💪 You've already completed ${completedTasks} tasks — great job! Keep the momentum going!\n\n`;
      }
      
      greetingMessage += `✨ Are you ready to crush more goals today?`;
      
      // Get task context for detailed workspace information
      const taskContext = await this.getUserTaskContext(userId);

      res.json({
        success: true,
        data: {
          greeting: greetingMessage,
          userName: user.name,
          timestamp: new Date(),
          taskSummary,
          urgentTasks: urgentTasks.slice(0, 3), // Show top 3 urgent tasks
          suggestions,
          taskContext // Include workspace data for detailed task information
        }
      });

    } catch (error) {
      console.error('Greeting error:', error);
      res.status(500).json({
        success: false,
        message: 'Error generating greeting',
        error: error.message
      });
    }
  }

  // Parse a due date from free text (supports English and Arabic basics)
  parseDueDateText(text) {
    if (!text || typeof text !== 'string') return null;
    const t = text.trim().toLowerCase();
    // Natural words
    if (t === 'today' || t === 'اليوم') {
      const d = new Date(); d.setHours(0,0,0,0); return d;
    }
    if (t === 'tomorrow' || t === 'بكرة' || t === 'غداً' || t === 'غدا') {
      const d = new Date(); d.setDate(d.getDate() + 1); d.setHours(0,0,0,0); return d;
    }
    if (t === 'بعد بكرة') {
      const d = new Date(); d.setDate(d.getDate() + 2); d.setHours(0,0,0,0); return d;
    }
    // ISO and common date formats
    const direct = new Date(text);
    if (!isNaN(direct.getTime())) return direct;
    // Try DD/MM/YYYY
    const m = t.match(/^([0-3]?\d)[\/\-]([0-1]?\d)[\/\-](\d{4})$/);
    if (m) {
      const dd = parseInt(m[1], 10);
      const mm = parseInt(m[2], 10) - 1;
      const yyyy = parseInt(m[3], 10);
      const d = new Date(yyyy, mm, dd);
      if (!isNaN(d.getTime())) return d;
    }
    // Try YYYY-MM-DD
    const iso = t.match(/^(\d{4})[\-\/]([0-1]?\d)[\-\/]([0-3]?\d)$/);
    if (iso) {
      const yyyy = parseInt(iso[1], 10);
      const mm = parseInt(iso[2], 10) - 1;
      const dd = parseInt(iso[3], 10);
      const d = new Date(yyyy, mm, dd);
      if (!isNaN(d.getTime())) return d;
    }
    return null;
  }

  // Helper method to get time of day greeting
  getTimeOfDay() {
    const hour = new Date().getHours();
    if (hour < 12) return 'صباح الخير';
    if (hour < 17) return 'مساء الخير';
    return 'مساء الخير';
  }

  // Get chat history
  async getChatHistory(req, res) {
    try {
      const userId = req.user.id;
      const { limit = 20 } = req.query;

      const history = await TitiChatHistory.getRecentHistory(userId, parseInt(limit));

      res.json({
        success: true,
        data: {
          messages: history,
          count: history.length
        }
      });

    } catch (error) {
      console.error('Chat history error:', error);
      res.status(500).json({
        success: false,
        message: 'Error fetching chat history',
        error: error.message
      });
    }
  }

  // Clear chat history (for new chat functionality)
  async clearChatHistory(req, res) {
    try {
      const userId = req.user.id;

      await TitiChatHistory.deleteOne({ userId });

      res.json({
        success: true,
        message: 'Chat history cleared successfully'
      });

    } catch (error) {
      console.error('Clear chat history error:', error);
      res.status(500).json({
        success: false,
        message: 'Error clearing chat history',
        error: error.message
      });
    }
  }
}

module.exports = {
  chat: async (req, res) => {
    const controller = new TitiController();
    return controller.chat(req, res);
  },
  getGreeting: async (req, res) => {
    const controller = new TitiController();
    return controller.getGreeting(req, res);
  },
  getChatHistory: async (req, res) => {
    const controller = new TitiController();
    return controller.getChatHistory(req, res);
  },
  clearChatHistory: async (req, res) => {
    const controller = new TitiController();
    return controller.clearChatHistory(req, res);
  }
};