const Event = require('../models/Event');
const Sections = require('../models/Sections');
const GroupTask = require('../models/GroupTask');
const GroupMembership = require('../models/GroupMembership');
const moment = require('moment');

// Get all calendar data (events + tasks) for a user
exports.getCalendarData = async (req, res) => {
    try {
        const userId = req.user._id;
        const { workspaceId, startDate, endDate } = req.query;

        console.log('Calendar Debug - User ID:', userId);
        console.log('Calendar Debug - User object:', req.user);
        console.log('Calendar Debug - Query params:', { workspaceId, startDate, endDate });

        // Date range for filtering (default to current month if not provided)
        const start = startDate ? new Date(startDate) : new Date(new Date().getFullYear(), new Date().getMonth(), 1);
        const end = endDate ? new Date(endDate) : new Date(new Date().getFullYear(), new Date().getMonth() + 1, 0);

        // Get user's custom events
        let eventsQuery = { userId };
        if (workspaceId) {
            eventsQuery.workspaceId = workspaceId;
        }

        const events = await Event.find({
            ...eventsQuery,
            startDate: { $gte: start, $lte: end }
        }).sort({ startDate: 1 });

        // Get user's personal tasks with due dates
        let sectionsQuery = { userId };
        if (workspaceId) {
            sectionsQuery.workspace = workspaceId;
        }

        console.log('Calendar Debug - Sections query:', sectionsQuery);

        const sections = await Sections.find(sectionsQuery);
        const personalTasks = [];

        console.log('Calendar Debug - Found sections:', sections.length);
        console.log('Calendar Debug - Date range:', { start, end });

        sections.forEach(section => {
            console.log(`Calendar Debug - Section "${section.name}" has ${section.tasks.length} tasks`);
            section.tasks.forEach(task => {
                console.log(`Calendar Debug - Task "${task.name}" dueDate:`, task.dueDate);
                if (task.dueDate && task.dueDate >= start && task.dueDate <= end) {
                    personalTasks.push({
                        _id: task._id,
                        title: task.name,
                        description: task.description,
                        startDate: task.dueDate,
                        endDate: task.dueDate,
                        allDay: true,
                        type: 'task',
                        priority: task.priority,
                        isImportant: task.isImportant,
                        isDone: task.isDone,
                        tags: task.tags,
                        sectionId: section._id,
                        sectionName: section.name,
                        color: task.isDone ? '#6B7280' : (task.isImportant ? '#EF4444' : '#3B82F6') // Gray-500 if done, red-500 if important, blue-500 otherwise
                    });
                }
            });
        });

        // Get user's group tasks
        const userEmail = req.user.email;
        const groupMemberships = await GroupMembership.find({
            $or: [
                { user: userId },
                { email: userEmail }
            ]
        }).populate('group');

        const groupTasks = [];
        for (const membership of groupMemberships) {
            const tasks = await GroupTask.find({
                group: membership.group._id,
                dueDate: { $gte: start, $lte: end },
                $or: [
                    { assignedTo: userEmail },
                    { createdBy: userId }
                ]
            });

            tasks.forEach(task => {
                if (task.dueDate) {
                    groupTasks.push({
                        _id: task._id,
                        title: task.title,
                        description: task.description,
                        startDate: task.dueDate,
                        endDate: task.dueDate,
                        allDay: true,
                        type: 'group-task',
                        priority: task.priority,
                        completed: task.completed,
                        assignedTo: task.assignedTo,
                        groupId: membership.group._id,
                        groupName: membership.group.name,
                        color: task.completed ? '#6B7280' : '#8B5CF6' // Gray-500 if completed, purple-500 for group tasks
                    });
                }
            });
        }

        // Combine all calendar items
        const calendarItems = [
            ...events.map(event => ({
                _id: event._id,
                title: event.title,
                description: event.description,
                startDate: event.startDate,
                endDate: event.endDate,
                allDay: event.allDay,
                type: event.type,
                color: event.color,
                reminder: event.reminder
            })),
            ...personalTasks,
            ...groupTasks
        ];

        console.log('Calendar Debug - Final calendar items count:', calendarItems.length);
        console.log('Calendar Debug - Personal tasks count:', personalTasks.length);
        console.log('Calendar Debug - Group tasks count:', groupTasks.length);
        console.log('Calendar Debug - Events count:', events.length);

        // Sort by start date
        calendarItems.sort((a, b) => new Date(a.startDate) - new Date(b.startDate));

        // Group by date for easier frontend consumption
        const groupedByDate = {};
        calendarItems.forEach(item => {
            const dateKey = new Date(item.startDate).toISOString().split('T')[0];
            if (!groupedByDate[dateKey]) {
                groupedByDate[dateKey] = [];
            }
            groupedByDate[dateKey].push(item);
        });

        res.json({
            success: true,
            data: {
                items: calendarItems,
                groupedByDate,
                summary: {
                    totalEvents: events.length,
                    totalPersonalTasks: personalTasks.length,
                    totalGroupTasks: groupTasks.length,
                    totalItems: calendarItems.length
                }
            }
        });

    } catch (error) {
        console.error('Error fetching calendar data:', error);
        res.status(500).json({
            success: false,
            message: 'Failed to fetch calendar data',
            error: error.message
        });
    }
};

// Create a new event
exports.createEvent = async (req, res) => {
    try {
        const userId = req.user._id;
        const { title, description, startDate, endDate, allDay, workspaceId, reminder, color } = req.body;

        // Validation
        if (!title || !startDate) {
            return res.status(400).json({
                success: false,
                message: 'Title and start date are required'
            });
        }

        // Handle workspaceId - if not provided, find or create default workspace
        let finalWorkspaceId = workspaceId;
        if (!finalWorkspaceId) {
            const Workspace = require('../models/Workspace');

            // Try to find user's default workspace
            let defaultWorkspace = await Workspace.findOne({ userId, isDefault: true });

            // If no default workspace exists, create one
            if (!defaultWorkspace) {
                defaultWorkspace = new Workspace({
                    name: "Home",
                    userId,
                    isDefault: true
                });
                await defaultWorkspace.save();
            }

            finalWorkspaceId = defaultWorkspace._id;
        }

        const eventData = {
            title: title.trim(),
            description: description?.trim() || '',
            startDate: new Date(startDate),
            endDate: endDate ? new Date(endDate) : new Date(startDate),
            allDay: allDay || false,
            userId,
            workspaceId: finalWorkspaceId,
            reminder: reminder || { enabled: false, minutes: 10 },
            color: color || '#10B981'
        };

        const event = new Event(eventData);
        await event.save();

        res.status(201).json({
            success: true,
            message: 'Event created successfully',
            data: event
        });

    } catch (error) {
        console.error('Error creating event:', error);
        res.status(500).json({
            success: false,
            message: 'Failed to create event',
            error: error.message
        });
    }
};

// Update an event
exports.updateEvent = async (req, res) => {
    try {
        const { eventId } = req.params;
        const userId = req.user._id;
        const updates = req.body;

        const event = await Event.findOne({ _id: eventId, userId });
        if (!event) {
            return res.status(404).json({
                success: false,
                message: 'Event not found'
            });
        }

        // Update allowed fields
        const allowedUpdates = ['title', 'description', 'startDate', 'endDate', 'allDay', 'reminder', 'color'];
        allowedUpdates.forEach(field => {
            if (updates[field] !== undefined) {
                if (field === 'startDate' || field === 'endDate') {
                    event[field] = new Date(updates[field]);
                } else {
                    event[field] = updates[field];
                }
            }
        });

        await event.save();

        res.json({
            success: true,
            message: 'Event updated successfully',
            data: event
        });

    } catch (error) {
        console.error('Error updating event:', error);
        res.status(500).json({
            success: false,
            message: 'Failed to update event',
            error: error.message
        });
    }
};

// Delete an event
exports.deleteEvent = async (req, res) => {
    try {
        const { eventId } = req.params;
        const userId = req.user._id;

        const event = await Event.findOneAndDelete({ _id: eventId, userId });
        if (!event) {
            return res.status(404).json({
                success: false,
                message: 'Event not found'
            });
        }

        res.json({
            success: true,
            message: 'Event deleted successfully'
        });

    } catch (error) {
        console.error('Error deleting event:', error);
        res.status(500).json({
            success: false,
            message: 'Failed to delete event',
            error: error.message
        });
    }
};

// Get events for a specific date
exports.getEventsForDate = async (req, res) => {
    try {
        const userId = req.user._id;
        const { date, workspaceId } = req.query;

        if (!date) {
            return res.status(400).json({
                success: false,
                message: 'Date parameter is required'
            });
        }

        const targetDate = new Date(date);
        const startOfDay = new Date(targetDate.setHours(0, 0, 0, 0));
        const endOfDay = new Date(targetDate.setHours(23, 59, 59, 999));

        // Get events for the date
        let eventsQuery = {
            userId,
            startDate: { $gte: startOfDay, $lte: endOfDay }
        };
        if (workspaceId) {
            eventsQuery.workspaceId = workspaceId;
        }

        const events = await Event.find(eventsQuery).sort({ startDate: 1 });

        // Get tasks for the date
        let sectionsQuery = { userId };
        if (workspaceId) {
            sectionsQuery.workspaceId = workspaceId;
        }

        const sections = await Sections.find(sectionsQuery);
        const tasksForDate = [];

        sections.forEach(section => {
            section.tasks.forEach(task => {
                if (task.dueDate) {
                    const taskDate = new Date(task.dueDate);
                    if (taskDate >= startOfDay && taskDate <= endOfDay) {
                        tasksForDate.push({
                            _id: task._id,
                            title: task.name,
                            description: task.description,
                            dueDate: task.dueDate,
                            type: 'task',
                            priority: task.priority,
                            isImportant: task.isImportant,
                            isDone: task.isDone,
                            sectionId: section._id,
                            sectionName: section.name
                        });
                    }
                }
            });
        });

        res.json({
            success: true,
            data: {
                events,
                tasks: tasksForDate,
                date: date
            }
        });

    } catch (error) {
        console.error('Error fetching events for date:', error);
        res.status(500).json({
            success: false,
            message: 'Failed to fetch events for date',
            error: error.message
        });
    }
};

// Get today's tasks and events for notifications
exports.getTodayItems = async (req, res) => {
    try {
        console.log('=== getTodayItems API called ===');
        console.log('User object:', req.user);

        const userId = req.user._id || req.user.id;
        const userEmail = req.user.email;
        const { workspaceId, tzOffset } = req.query;

        console.log('Workspace ID from query:', workspaceId);

        // Get today's date range using moment; tzOffset is minutes from local to UTC (Date.getTimezoneOffset)
        // The frontend sends Date.getTimezoneOffset() * -1.
        // Example: UTC+2 is -120. Frontend sends 120.
        // moment.utcOffset() expects the offset in minutes FROM UTC (e.g., +120 for UTC+2).
        // So we use the value directly.
        const rawOffset = Number.isFinite(Number(tzOffset)) ? Number(tzOffset) : 0;
        const localOffset = rawOffset;

        const today = moment().utcOffset(localOffset);
        const startOfDay = today.clone().startOf('day').toDate();
        const endOfDay = today.clone().endOf('day').toDate();

        console.log('Timezone Debug:', {
            browserOffset: rawOffset,
            momentOffset: localOffset,
            serverTime: new Date(),
            calculatedStart: startOfDay,
            calculatedEnd: endOfDay
        });

        // Get today's events (include events that overlap today)
        let eventsQuery = { userId };
        if (workspaceId) {
            eventsQuery.workspaceId = workspaceId;
        }

        const todayEventsRaw = await Event.find({
            ...eventsQuery,
            $and: [
                { startDate: { $lte: endOfDay } },
                { endDate: { $gte: startOfDay } }
            ]
        }).sort({ startDate: 1 });

        // Normalize event payload for UI (add type for consistent coloring)
        const todayEvents = todayEventsRaw.map(e => ({
            _id: e._id,
            title: e.title,
            description: e.description,
            startDate: e.startDate,
            endDate: e.endDate,
            allDay: e.allDay,
            color: e.color,
            type: 'event'
        }));

        // Get today's tasks - include sections owned by the user OR containing tasks assigned to the user
        // If workspaceId is provided, also require section.workspace to match
        let sectionsQuery = { $or: [{ userId }, { 'tasks.assignedTo.email': userEmail }] };
        if (workspaceId) {
            sectionsQuery.workspace = workspaceId;
        }
        const sections = await Sections.find(sectionsQuery);
        const personalTodayTasks = [];
        const overdueTasks = [];

        console.log('Today date range:', { startOfDay, endOfDay });
        console.log('Found sections:', sections.length);

        sections.forEach(section => {
            // console.log(`Processing section: ${section.name}, tasks: ${section.tasks.length}`);
            section.tasks.forEach(task => {
                if (task.dueDate && !task.isDone) {
                    const taskMoment = moment(task.dueDate).utcOffset(localOffset);
                    const isOwnSection = String(section.userId) === String(userId);
                    const isAssignedToUser = Array.isArray(task.assignedTo) && task.assignedTo.some(a => (a.email || '').toLowerCase() === (userEmail || '').toLowerCase());

                    // Only include tasks the user owns or is assigned to
                    if (!isOwnSection && !isAssignedToUser) {
                        return;
                    }
                    // console.log(`Task: ${task.name}, dueDate: ${task.dueDate}, taskMoment: ${taskMoment.toDate()}, isDone: ${task.isDone}`);

                    // Inclusive range check to avoid timezone mismatches
                    if (taskMoment.isBetween(moment(startOfDay), moment(endOfDay), undefined, '[]')) {
                        // console.log(`Adding task to today: ${task.name}`);
                        personalTodayTasks.push({
                            _id: task._id,
                            title: task.name,
                            description: task.description,
                            dueDate: task.dueDate,
                            priority: task.priority,
                            isImportant: task.isImportant,
                            sectionName: section.name,
                            type: 'personalTask'
                        });
                    } else if (taskMoment.isBefore(moment(startOfDay))) {
                        // console.log(`Adding task to overdue: ${task.name}`);
                        overdueTasks.push({
                            _id: task._id,
                            title: task.name,
                            description: task.description,
                            dueDate: task.dueDate,
                            priority: task.priority,
                            isImportant: task.isImportant,
                            sectionName: section.name,
                            isOverdue: true,
                            type: 'personalTask'
                        });
                    }
                }
            });
        });

        // Add group tasks (assigned to the user or created by the user)
        console.log('Fetching group tasks for today and overdue');

        // Case-insensitive email matching regex
        const emailRegex = new RegExp(`^${userEmail}$`, 'i');

        const groupTodayQuery = {
            $and: [
                { dueDate: { $ne: null } },
                { dueDate: { $gte: startOfDay, $lte: endOfDay } },
                { completed: false }
            ],
            $or: [
                { assignedTo: { $in: [emailRegex] } }, // Check if array contains email (case-insensitive)
                { createdBy: userId }
            ]
        };
        const groupOverdueQuery = {
            $and: [
                { dueDate: { $ne: null } },
                { dueDate: { $lt: startOfDay } },
                { completed: false }
            ],
            $or: [
                { assignedTo: { $in: [emailRegex] } },
                { createdBy: userId }
            ]
        };

        const [groupTodayTasksRaw, groupOverdueTasksRaw] = await Promise.all([
            GroupTask.find(groupTodayQuery).populate('group', 'name'),
            GroupTask.find(groupOverdueQuery).populate('group', 'name')
        ]);
        const groupTodayTasks = groupTodayTasksRaw.map(gt => ({
            _id: gt._id,
            title: gt.title,
            description: gt.description,
            dueDate: gt.dueDate,
            priority: gt.priority,
            isImportant: false,
            sectionName: gt.group?.name || 'Group',
            groupName: gt.group?.name || 'Group',
            type: 'groupTask'
        }));

        const groupOverdueTasks = groupOverdueTasksRaw.map(gt => ({
            _id: gt._id,
            title: gt.title,
            description: gt.description,
            dueDate: gt.dueDate,
            priority: gt.priority,
            isImportant: false,
            sectionName: gt.group?.name || 'Group',
            groupName: gt.group?.name || 'Group',
            isOverdue: true,
            type: 'groupTask'
        }));

        // Add group overdue tasks to combined overdue list
        overdueTasks.push(...groupOverdueTasks);

        console.log('Today tasks found (personal):', personalTodayTasks.length);
        console.log('Today tasks found (group):', groupTodayTasks.length);
        console.log('Overdue tasks found (including group):', overdueTasks.length);

        const totalTasks = personalTodayTasks.length + groupTodayTasks.length;
        const totalEvents = todayEvents.length;
        const totalItems = totalTasks + totalEvents;

        res.json({
            success: true,
            data: {
                events: todayEvents,
                personalTasks: personalTodayTasks,
                groupTasks: groupTodayTasks,
                overdueTasks: overdueTasks,
                summary: {
                    totalEvents: totalEvents,
                    totalTasks: totalTasks,
                    totalPersonalTasks: personalTodayTasks.length,
                    totalGroupTasks: groupTodayTasks.length,
                    totalOverdueTasks: overdueTasks.length,
                    totalItems: totalItems,
                    message: `You have ${totalTasks} task${totalTasks !== 1 ? 's' : ''} and ${totalEvents} event${totalEvents !== 1 ? 's' : ''} today. ${overdueTasks.length > 0 ? `${overdueTasks.length} overdue task${overdueTasks.length !== 1 ? 's' : ''}.` : ''}`
                }
            }
        });

    } catch (error) {
        console.error('Error fetching today items:', error);
        res.status(500).json({
            success: false,
            message: 'Failed to fetch today items',
            error: error.message
        });
    }
};