const User = require('../models/User');
const Analytics = require('../models/Analytics');
const Workspace = require('../models/Workspace');
const Section = require('../models/Sections');
const AutomationReport = require('../models/AutomationReport');
const axios = require('axios');
const phoneUtils = require('../utils/phoneUtils');

// WhatsApp sender config (reuse existing envs)
const WASENDER_URL = process.env.WASENDER_URL || 'https://wasenderapi.com/api/send-message';
const WASENDER_API_TOKEN = process.env.WASENDER_API_TOKEN || '30ddf4326470551c5ff032627761b0736d5981e304cb1c5a853bbb99fddac45d';
const ADMIN_BROADCAST_DELAY_MS = Number(process.env.ADMIN_BROADCAST_DELAY_MS || process.env.TITI_AUTOMATION_DELAY_MS || 2000);
const PLACEHOLDER_PHONES = (process.env.TITI_AUTOMATION_PLACEHOLDER_PHONES || '+201000000001,+201000000002')
  .split(',').map(s => s.trim()).filter(Boolean);

// In-memory broadcast jobs store for progress tracking
const broadcastJobs = new Map();
function makeJobId() { return `${Date.now()}-${Math.random().toString(36).slice(2, 10)}`; }

function sleep(ms) { return new Promise(res => setTimeout(res, ms)); }
function sanitize(text) { return String(text || '').replace(/`/g, '').trim(); }
// Normalize Egyptian phone using shared util, with graceful fallback
function normalizePhone(input) {
  try {
    const normalizer = (typeof phoneUtils === 'function')
      ? phoneUtils
      : (phoneUtils && typeof phoneUtils.normalizeEgyptPhone === 'function')
        ? phoneUtils.normalizeEgyptPhone
        : null;
    return normalizer ? normalizer(input) : String(input || '').trim();
  } catch (_) {
    return String(input || '').trim();
  }
}

// Pre-normalize placeholder phones for reliable matching
const NORMALIZED_PLACEHOLDER_SET = new Set(
  PLACEHOLDER_PHONES.map(p => normalizePhone(p)).filter(Boolean)
);

function isPlaceholderPhone(to) {
  const normalized = normalizePhone(to);
  return normalized ? NORMALIZED_PLACEHOLDER_SET.has(normalized) : false;
}
function isValidEgyptPhone(to) {
  const digits = String(to).replace(/\D/g, '');
  const msisdn = digits.startsWith('20') ? digits.slice(2) : digits;
  try {
    return !!phoneUtils.isValidEgyptMobile && phoneUtils.isValidEgyptMobile(msisdn);
  } catch (_) {
    return true; // fallback: assume valid
  }
}
function personalize(text, name) {
  const displayName = name || 'صديق تيتي';
  return String(text || '')
    .replace(/\{\{\s*name\s*\}\}/gi, displayName)
    .replace(/\{\{\s*اسمك\s*\}\}/gi, displayName);
}
async function sendWhatsApp(to, text) {
  const payload = { to, text: sanitize(text) };
  try {
    const res = await axios.post(WASENDER_URL, payload, {
      headers: {
        'Content-Type': 'application/json',
        'Authorization': `Bearer ${WASENDER_API_TOKEN}`
      },
      timeout: 20000
    });
    return res.data;
  } catch (err) {
    const status = err.response?.status;
    const body = err.response?.data;
    // Handle rate limiting: retry once after waiting suggested time
    if (status === 429) {
      const retryAfterSec = Number(body?.retry_after) || 5; // default 5s
      const waitMs = Math.max(ADMIN_BROADCAST_DELAY_MS, retryAfterSec * 1000);
      await sleep(waitMs);
      const res2 = await axios.post(WASENDER_URL, payload, {
        headers: {
          'Content-Type': 'application/json',
          'Authorization': `Bearer ${WASENDER_API_TOKEN}`
        },
        timeout: 20000
      });
      return res2.data;
    }
    throw err;
  }
}

// Map an error to a human readable reason for admin visibility
function formatErrorReason(err) {
  try {
    const status = err?.response?.status;
    const code = err?.code;
    if (status === 401 || status === 403) return 'Authorization error with WhatsApp API';
    if (status === 400) return 'Invalid request or phone format';
    if (status === 404) return 'WhatsApp API endpoint not found';
    if (status === 429) return 'Rate limited by WhatsApp API';
    if (status >= 500) return 'WhatsApp API server error';
    if (code === 'ECONNABORTED') return 'Network timeout while sending message';
    return err?.message || 'Unknown error while sending message';
  } catch (_) {
    return 'Unknown error while sending message';
  }
}

// Summarize latest status per recipient from a job results array
function summarizeRecipientStatuses(job) {
  const map = new Map();
  for (const r of job.results || []) {
    const key = r.to;
    if (!key) continue;
    map.set(key, { to: r.to, status: r.status || (r.dryRun ? 'dryRun' : 'failed'), reason: r.reason, code: r.code, attempts: (map.get(key)?.attempts || 0) + 1 });
  }
  return Array.from(map.values());
}

// Get dashboard analytics
exports.getDashboardStats = async (req, res) => {
    try {
        let analytics = await Analytics.findOne();
        if (!analytics) {
            analytics = await Analytics.create({});
        }

        // Update analytics in real-time
        const [totalUsers, activeUsers, totalWorkspaces, totalSections] = await Promise.all([
            User.countDocuments(),
            User.countDocuments({ status: 'active' }),
            Workspace.countDocuments(),
            Section.countDocuments()
        ]);

        // Calculate task statistics
        const sections = await Section.find();
        let totalTasks = 0;
        let completedTasks = 0;

        sections.forEach(section => {
            section.tasks.forEach(task => {
                totalTasks++;
                if (task.isDone) completedTasks++;
            });
        });

        // Update user role distribution
        const usersByRole = await User.aggregate([
            { $group: { _id: '$role', count: { $sum: 1 } } }
        ]);

        const roleDistribution = {
            admin: 0,
            solo: 0,
            team: 0,
            company: 0
        };

        usersByRole.forEach(role => {
            roleDistribution[role._id] = role.count;
        });

        // Update analytics
        analytics.totalUsers = totalUsers;
        analytics.activeUsers = activeUsers;
        analytics.totalWorkspaces = totalWorkspaces;
        analytics.totalSections = totalSections;
        analytics.totalTasks = totalTasks;
        analytics.completedTasks = completedTasks;
        analytics.usersByRole = roleDistribution;
        analytics.taskCompletionRate = totalTasks > 0 ? (completedTasks / totalTasks) * 100 : 0;
        analytics.averageTasksPerUser = totalUsers > 0 ? totalTasks / totalUsers : 0;
        analytics.lastUpdated = new Date();

        await analytics.save();

        res.json(analytics);
    } catch (error) {
        res.status(500).json({ message: 'Error fetching dashboard statistics', error: error.message });
    }
};

// Get user management data
exports.getUsers = async (req, res) => {
    try {
        const users = await User.find().select('-password -otp');
        res.json(users);
    } catch (error) {
        res.status(500).json({ message: 'Error fetching users', error: error.message });
    }
};

// Update user status
exports.updateUserStatus = async (req, res) => {
    try {
        const { userId } = req.params;
        const { status } = req.body;
        const user = await User.findByIdAndUpdate(
            userId,
            { status },
            { new: true }
        ).select('-password -otp');

        if (!user) {
            return res.status(404).json({ message: 'User not found' });
        }

        res.json(user);
    } catch (error) {
        res.status(500).json({ message: 'Error updating user status', error: error.message });
    }
};

// Get system activity logs
exports.getActivityLogs = async (req, res) => {
    try {
        const { startDate, endDate } = req.query;
        const query = {};

        if (startDate && endDate) {
            query.createdAt = {
                $gte: new Date(startDate),
                $lte: new Date(endDate)
            };
        }

        const dailyActiveUsers = await User.aggregate([
            { $match: { lastLogin: { $exists: true, ...query.createdAt } } },
            {
                $group: {
                    _id: { $dateToString: { format: '%Y-%m-%d', date: '$lastLogin' } },
                    count: { $sum: 1 }
                }
            },
            { $sort: { '_id': -1 } }
        ]);

        res.json(dailyActiveUsers);
    } catch (error) {
        res.status(500).json({ message: 'Error fetching activity logs', error: error.message });
    }
};

// Advanced analytics: section/task breakdown, login trends, most active users, error rates
exports.getAdvancedAnalytics = async (req, res) => {
    try {
        // Section/task breakdown
        const sections = await Section.find();
        const sectionBreakdown = sections.map(section => ({
            sectionName: section.name,
            totalTasks: section.tasks.length,
            completedTasks: section.tasks.filter(t => t.isDone).length,
            overdueTasks: section.tasks.filter(t => t.dueDate && !t.isDone && t.dueDate < new Date()).length
        }));

        // User growth (last 30 days) with fallback to ObjectId timestamp when createdAt is missing
        const today = new Date();
        const lastMonth = new Date();
        lastMonth.setDate(today.getDate() - 29);
        const userGrowth = await User.aggregate([
            { $addFields: { createdDate: { $ifNull: ['$createdAt', { $toDate: '$_id' }] } } },
            { $match: { createdDate: { $gte: lastMonth } } },
            {
                $group: {
                    _id: { $dateToString: { format: '%Y-%m-%d', date: '$createdDate' } },
                    count: { $sum: 1 }
                }
            },
            { $sort: { '_id': 1 } }
        ]);

        // Login trends (last 30 days)
        const loginTrends = await User.aggregate([
            { $match: { lastLogin: { $gte: lastMonth } } },
            {
                $group: {
                    _id: { $dateToString: { format: '%Y-%m-%d', date: '$lastLogin' } },
                    count: { $sum: 1 }
                }
            },
            { $sort: { '_id': 1 } }
        ]);

        // Task growth (last 30 days)
        const taskGrowth = await Section.aggregate([
            { $unwind: '$tasks' },
            { $match: { 'tasks.createdAt': { $gte: lastMonth } } },
            {
                $group: {
                    _id: { $dateToString: { format: '%Y-%m-%d', date: '$tasks.createdAt' } },
                    count: { $sum: 1 }
                }
            },
            { $sort: { '_id': 1 } }
        ]);

        // Task completion by day (approx using updatedAt when isDone)
        const taskCompletionByDay = await Section.aggregate([
            { $unwind: '$tasks' },
            { $match: { 'tasks.isDone': true, 'tasks.updatedAt': { $gte: lastMonth } } },
            {
                $group: {
                    _id: { $dateToString: { format: '%Y-%m-%d', date: '$tasks.updatedAt' } },
                    count: { $sum: 1 }
                }
            },
            { $sort: { '_id': 1 } }
        ]);

        // Task priority distribution
        const taskPriorityDistribution = await Section.aggregate([
            { $unwind: '$tasks' },
            {
                $group: {
                    _id: '$tasks.priority',
                    count: { $sum: 1 }
                }
            },
            { $project: { _id: 0, priority: '$_id', count: 1 } },
            { $sort: { priority: 1 } }
        ]);

        // Most active users (by login count)
        const mostActiveUsers = await User.find().sort({ loginCount: -1 }).limit(5).select('name email loginCount');
        // Least active users (by login count)
        const leastActiveUsers = await User.find().sort({ loginCount: 1 }).limit(5).select('name email loginCount');

        // Most completed tasks (users)
        const users = await User.find();
        const userTaskStats = await Promise.all(users.map(async user => {
            const userSections = await Section.find({ userId: user._id });
            let completed = 0, total = 0, overdue = 0;
            userSections.forEach(section => {
                section.tasks.forEach(task => {
                    total++;
                    if (task.isDone) completed++;
                    if (task.dueDate && !task.isDone && task.dueDate < new Date()) overdue++;
                });
            });
            return { userId: user._id, name: user.name, email: user.email, completed, total, overdue };
        }));
        const mostCompletedTasks = [...userTaskStats].sort((a, b) => b.completed - a.completed).slice(0, 5);
        const mostOverdueTasks = [...userTaskStats].sort((a, b) => b.overdue - a.overdue).slice(0, 5);

        // Section/task distribution
        const sectionDistribution = sections.map(section => ({
            sectionName: section.name,
            taskCount: section.tasks.length
        }));

        // Overdue buckets across all tasks
        const overdueBuckets = { '0-7': 0, '8-30': 0, '31+': 0 };
        for (const section of sections) {
            for (const task of section.tasks) {
                if (task.dueDate && !task.isDone && task.dueDate < today) {
                    const days = Math.floor((today - task.dueDate) / (1000 * 60 * 60 * 24));
                    if (days <= 7) overdueBuckets['0-7']++;
                    else if (days <= 30) overdueBuckets['8-30']++;
                    else overdueBuckets['31+']++;
                }
            }
        }
        const overdueDistribution = Object.entries(overdueBuckets).map(([bucket, count]) => ({ bucket, count }));

        // Top workspaces by task count
        const tasksPerWorkspace = await Section.aggregate([
            { $project: { workspace: 1, taskCount: { $size: '$tasks' } } },
            { $group: { _id: '$workspace', taskCount: { $sum: '$taskCount' } } },
            { $sort: { taskCount: -1 } },
            { $limit: 10 }
        ]);
        const workspaceIds = tasksPerWorkspace.map(x => x._id).filter(Boolean);
        const workspaces = await Workspace.find({ _id: { $in: workspaceIds } }).select('_id name');
        const wsMap = new Map(workspaces.map(w => [String(w._id), w.name]));
        const topWorkspacesByTasks = tasksPerWorkspace.map(x => ({
            workspaceId: x._id,
            workspaceName: wsMap.get(String(x._id)) || 'Unassigned',
            taskCount: x.taskCount
        }));

        // User status distribution
        const userStatusDistribution = await User.aggregate([
            { $group: { _id: '$status', count: { $sum: 1 } } },
            { $project: { _id: 0, status: '$_id', count: 1 } }
        ]);

        // Recent user signups (last 7 days)
        const recentSignups = await User.find({ createdAt: { $gte: new Date(Date.now() - 7*24*60*60*1000) } }).select('name email createdAt');

        // Broadcast failure reasons and skipped phone issues from in-memory jobs
        const failureCounts = {};
        const skipCounts = {};
        for (const job of broadcastJobs.values()) {
            for (const r of job.results || []) {
                if (r.status === 'failed') {
                    const reason = r.reason || 'Unknown';
                    failureCounts[reason] = (failureCounts[reason] || 0) + 1;
                }
            }
            for (const s of job.skipped || []) {
                const reason = s.reason || 'unknown';
                skipCounts[reason] = (skipCounts[reason] || 0) + 1;
            }
        }
        const failureReasons = Object.entries(failureCounts).map(([reason, count]) => ({ reason, count })).sort((a, b) => b.count - a.count).slice(0, 10);
        const skippedPhoneIssues = Object.entries(skipCounts).map(([reason, count]) => ({ reason, count })).sort((a, b) => b.count - a.count);

        res.json({
            sectionBreakdown,
            userGrowth,
            loginTrends,
            taskGrowth,
            taskCompletionByDay,
            taskPriorityDistribution,
            mostActiveUsers,
            leastActiveUsers,
            mostCompletedTasks,
            mostOverdueTasks,
            sectionDistribution,
            overdueDistribution,
            topWorkspacesByTasks,
            userStatusDistribution,
            recentSignups,
            failureReasons,
            skippedPhoneIssues
        });
    } catch (error) {
        res.status(500).json({ message: 'Error fetching advanced analytics', error: error.message });
    }
};

// Automation reports: list recent runs (optionally filtered by type)
exports.getAutomationReports = async (req, res) => {
  try {
    const { type, limit = 20 } = req.query;
    const q = {};
    if (type) q.type = type; // 'titi' | 'daily'
    const reports = await AutomationReport.find(q)
      .sort({ startedAt: -1 })
      .limit(Math.min(Number(limit) || 20, 100))
      .select('type runId dryRun total processed successCount failCount startedAt endedAt');
    res.json({ reports });
  } catch (error) {
    res.status(500).json({ message: 'Error fetching automation reports', error: error.message });
  }
};

// Automation report details by id (includes per-recipient attempts)
exports.getAutomationReportDetails = async (req, res) => {
  try {
    const { id } = req.params;
    const doc = await AutomationReport.findById(id);
    if (!doc) return res.status(404).json({ message: 'Report not found' });
    res.json(doc);
  } catch (error) {
    res.status(500).json({ message: 'Error fetching report details', error: error.message });
  }
};

// Control: force logout (set status to inactive)
exports.forceLogout = async (req, res) => {
    try {
        const { userId } = req.body;
        const user = await User.findByIdAndUpdate(userId, { status: 'inactive' }, { new: true });
        if (!user) return res.status(404).json({ message: 'User not found' });
        res.json({ message: 'User forced to logout', user });
    } catch (error) {
        res.status(500).json({ message: 'Error forcing logout', error: error.message });
    }
};

// Control: reset password (set a temp password)
exports.resetPassword = async (req, res) => {
    try {
        const { userId, tempPassword } = req.body;
        const bcrypt = require('bcryptjs');
        const hashed = await bcrypt.hash(tempPassword, 10);
        const user = await User.findByIdAndUpdate(userId, { password: hashed }, { new: true });
        if (!user) return res.status(404).json({ message: 'User not found' });
        res.json({ message: 'Password reset', user });
    } catch (error) {
        res.status(500).json({ message: 'Error resetting password', error: error.message });
    }
};

// Control: change user role
exports.changeUserRole = async (req, res) => {
    try {
        const { userId, newRole } = req.body;
        const user = await User.findByIdAndUpdate(userId, { role: newRole }, { new: true });
        if (!user) return res.status(404).json({ message: 'User not found' });
        res.json({ message: 'User role updated', user });
    } catch (error) {
        res.status(500).json({ message: 'Error changing user role', error: error.message });
    }
};

// Broadcast: send a custom message to all users (filtered)
exports.broadcastMessage = async (req, res) => {
    try {
        const { message, dryRun = false, roles = [], status = 'active', delayMs } = req.body;
        const text = sanitize(message);
        if (!text || !text.trim()) {
            return res.status(400).json({ message: 'Message text is required' });
        }

        // Build query for recipients
        const query = {};
        if (Array.isArray(roles) && roles.length) {
            query.role = { $in: roles };
        }
        if (status) {
            query.status = status;
        }

        const users = await User.find(query).select('name phone');
        const normalizer = (typeof phoneUtils === 'function')
            ? phoneUtils
            : (phoneUtils && typeof phoneUtils.normalizeEgyptPhone === 'function')
                ? phoneUtils.normalizeEgyptPhone
                : null;

        const targets = [];
        for (const u of users) {
            if (!u.phone) continue;
            let to = null;
            try {
                to = normalizer ? normalizer(u.phone) : String(u.phone).trim();
            } catch (_) {
                to = String(u.phone).trim();
            }
            if (!to) continue;
            if (isPlaceholderPhone(to)) continue;
            if (!isValidEgyptPhone(to)) continue;
            targets.push({ to, name: u.name });
        }

        const effectiveDelay = Number(delayMs ?? ADMIN_BROADCAST_DELAY_MS) || 0;
        let processed = 0, failed = 0;
        const results = [];

        for (let i = 0; i < targets.length; i++) {
            const { to, name } = targets[i];
            // Replace {{name}} tokens if present
            const personalized = personalize(text, name);
            if (dryRun) {
                results.push({ to, dryRun: true });
                processed++;
            } else {
                try {
                    await sendWhatsApp(to, personalized);
                    processed++;
                    results.push({ to, status: 'sent' });
                } catch (err) {
                    failed++;
                    results.push({ to, error: err.response?.status || err.message });
                }
            }
            if (i < targets.length - 1 && effectiveDelay > 0) {
                await sleep(effectiveDelay);
            }
        }

        return res.json({ total: targets.length, processed, failed, dryRun: !!dryRun, results: results.slice(0, 50) });
    } catch (error) {
        res.status(500).json({ message: 'Error broadcasting message', error: error.message });
    }
};

// Start a broadcast job and return a jobId for progress polling
exports.startBroadcastJob = async (req, res) => {
  try {
    const { message, dryRun = false, roles = [], status = 'active', delayMs } = req.body;
    const text = sanitize(message);
    if (!text || !text.trim()) {
      return res.status(400).json({ message: 'Message text is required' });
    }

    // Build query for recipients
    const query = {};
    if (Array.isArray(roles) && roles.length) {
      query.role = { $in: roles };
    }
    if (status) {
      query.status = status;
    }

    const users = await User.find(query).select('name phone _id');
    const normalizer = (typeof phoneUtils === 'function')
      ? phoneUtils
      : (phoneUtils && typeof phoneUtils.normalizeEgyptPhone === 'function')
        ? phoneUtils.normalizeEgyptPhone
        : null;

    const targets = [];
    const skipped = [];
    for (const u of users) {
      if (!u.phone) {
        skipped.push({ userId: u._id, name: u.name, reason: 'no_phone' });
        continue;
      }
      let to = null;
      try {
        to = normalizer ? normalizer(u.phone) : String(u.phone).trim();
      } catch (_) {
        to = String(u.phone).trim();
      }
      if (!to) {
        skipped.push({ userId: u._id, name: u.name, reason: 'empty_normalized_phone' });
        continue;
      }
      if (isPlaceholderPhone(to)) {
        skipped.push({ userId: u._id, name: u.name, to, reason: 'placeholder_phone' });
        continue;
      }
      if (!isValidEgyptPhone(to)) {
        skipped.push({ userId: u._id, name: u.name, to, reason: 'invalid_phone' });
        continue;
      }
      targets.push({ to, name: u.name });
    }

    const effectiveDelay = Number(delayMs ?? ADMIN_BROADCAST_DELAY_MS) || 0;
    const jobId = makeJobId();
    const job = {
      id: jobId,
      total: targets.length,
      processed: 0,
      failed: 0,
      dryRun: !!dryRun,
      status: 'running',
      delayMs: effectiveDelay,
      createdAt: new Date(),
      finishedAt: null,
      results: [],
      skipped,
      message: text,
      recipients: targets,
    };
    broadcastJobs.set(jobId, job);

    // Process asynchronously
    (async () => {
      try {
        let lastWasRateLimited = false;
        for (let i = 0; i < targets.length; i++) {
          // Support cancellation
          const current = broadcastJobs.get(jobId);
          if (!current || current.status !== 'running') break;

          const { to, name } = targets[i];
          const personalized = personalize(text, name);

          if (dryRun) {
            job.processed++;
            job.results.push({ to, status: 'dryRun' });
          } else {
            try {
              await sendWhatsApp(to, personalized);
              job.processed++;
              job.results.push({ to, status: 'sent' });
              lastWasRateLimited = false;
            } catch (err) {
              job.failed++;
              const reason = formatErrorReason(err);
              job.results.push({ to, status: 'failed', code: err.response?.status || null, reason });
              lastWasRateLimited = reason === 'Rate limited by WhatsApp API';
            }
          }

          // throttle
          if (i < targets.length - 1) {
            const base = effectiveDelay > 0 ? effectiveDelay : 0;
            const extra = lastWasRateLimited ? Math.max(base * 2, 6000) : 0;
            const wait = base + extra;
            if (wait > 0) {
              await sleep(wait);
            }
          }
        }
        const done = broadcastJobs.get(jobId);
        if (done && done.status === 'running') {
          done.status = 'completed';
          done.finishedAt = new Date();
        }
      } catch (err) {
        const failedJob = broadcastJobs.get(jobId);
        if (failedJob) {
          failedJob.status = 'error';
          failedJob.error = err.message || 'Broadcast error';
          failedJob.finishedAt = new Date();
        }
      }
    })();

    return res.json({ jobId, total: job.total, dryRun: job.dryRun, started: true });
  } catch (error) {
    res.status(500).json({ message: 'Error starting broadcast job', error: error.message });
  }
};

// Get progress for an existing broadcast job
exports.getBroadcastProgress = async (req, res) => {
  try {
    const { jobId } = req.params;
    const job = broadcastJobs.get(jobId);
    if (!job) {
      return res.status(404).json({ message: 'Job not found' });
    }
    const percent = job.total ? Math.round((job.processed / job.total) * 100) : 0;
    return res.json({
      jobId,
      total: job.total,
      processed: job.processed,
      failed: job.failed,
      dryRun: job.dryRun,
      status: job.status,
      percent,
      preview: job.results.slice(-10),
      error: job.error || null
    });
  } catch (error) {
    res.status(500).json({ message: 'Error fetching job progress', error: error.message });
  }
};

// Cancel an existing broadcast job
exports.cancelBroadcastJob = async (req, res) => {
  try {
    const { jobId } = req.body;
    const job = broadcastJobs.get(jobId);
    if (!job) {
      return res.status(404).json({ message: 'Job not found' });
    }
    job.status = 'cancelled';
    job.finishedAt = new Date();
    return res.json({ cancelled: true, jobId });
  } catch (error) {
    res.status(500).json({ message: 'Error cancelling job', error: error.message });
  }
};

// Return full results and skipped lists for a job for admin visibility
exports.getBroadcastResults = async (req, res) => {
  try {
    const { jobId } = req.params;
    const job = broadcastJobs.get(jobId);
    if (!job) {
      return res.status(404).json({ message: 'Job not found' });
    }
    const summary = summarizeRecipientStatuses(job);
    const failedList = summary.filter(s => s.status === 'failed');
    return res.json({
      jobId,
      total: job.total,
      processed: job.processed,
      failed: job.failed,
      status: job.status,
      results: job.results || [],
      skipped: job.skipped || [],
      failedList
    });
  } catch (error) {
    res.status(500).json({ message: 'Error fetching job results', error: error.message });
  }
};

// Retry failed recipients from a previous job
exports.retryBroadcastJob = async (req, res) => {
  try {
    const { jobId, delayMs, untilAllSuccess = false, maxRounds = 3 } = req.body;
    const original = broadcastJobs.get(jobId);
    if (!original) {
      return res.status(404).json({ message: 'Original job not found' });
    }
    const summary = summarizeRecipientStatuses(original);
    // Build fail list using latest statuses
    const failedRecipients = summary.filter(s => s.status === 'failed');
    if (!failedRecipients.length) {
      return res.json({ started: false, message: 'No failed recipients to retry' });
    }

    const effectiveDelay = Number(delayMs ?? original.delayMs ?? ADMIN_BROADCAST_DELAY_MS) || 0;
    const retryJobId = makeJobId();
    const retryJob = {
      id: retryJobId,
      total: failedRecipients.length,
      processed: 0,
      failed: 0,
      dryRun: false,
      status: 'running',
      delayMs: effectiveDelay,
      createdAt: new Date(),
      finishedAt: null,
      results: [],
      skipped: [],
      message: original.message,
      recipients: failedRecipients.map(fr => {
        const rec = (original.recipients || []).find(r => r.to === fr.to);
        return { to: fr.to, name: rec?.name };
      }),
      retryOf: jobId,
    };
    broadcastJobs.set(retryJobId, retryJob);

    // Process asynchronously, optionally multiple rounds until success
    (async () => {
      try {
        let round = 0;
        let pending = retryJob.recipients.slice();
        while (pending.length) {
          round++;
          let lastWasRateLimited = false;
          for (let i = 0; i < pending.length; i++) {
            const current = broadcastJobs.get(retryJobId);
            if (!current || current.status !== 'running') break;
            const { to, name } = pending[i];
            const personalized = personalize(retryJob.message, name);
            try {
              await sendWhatsApp(to, personalized);
              retryJob.processed++;
              retryJob.results.push({ to, status: 'sent', round });
              lastWasRateLimited = false;
            } catch (err) {
              retryJob.failed++;
              const reason = formatErrorReason(err);
              retryJob.results.push({ to, status: 'failed', code: err.response?.status || null, reason, round });
              lastWasRateLimited = reason === 'Rate limited by WhatsApp API';
            }
            if (i < pending.length - 1) {
              const base = effectiveDelay > 0 ? effectiveDelay : 0;
              const extra = lastWasRateLimited ? Math.max(base * 2, 6000) : 0;
              const wait = base + extra;
              if (wait > 0) {
                await sleep(wait);
              }
            }
          }
          if (!untilAllSuccess || round >= Number(maxRounds)) break;
          // Prepare next round with only those still failing
          const latest = summarizeRecipientStatuses(retryJob);
          pending = latest.filter(s => s.status === 'failed').map(s => {
            const rec = (retryJob.recipients || []).find(r => r.to === s.to);
            return { to: s.to, name: rec?.name };
          });
          if (pending.length && effectiveDelay > 0) {
            await sleep(effectiveDelay);
          }
        }
        const done = broadcastJobs.get(retryJobId);
        if (done && done.status === 'running') {
          done.status = 'completed';
          done.finishedAt = new Date();
        }
      } catch (err) {
        const failedJob = broadcastJobs.get(retryJobId);
        if (failedJob) {
          failedJob.status = 'error';
          failedJob.error = err.message || 'Retry broadcast error';
          failedJob.finishedAt = new Date();
        }
      }
    })();

    return res.json({ jobId: retryJobId, total: retryJob.total, started: true, retryOf: jobId });
  } catch (error) {
    res.status(500).json({ message: 'Error retrying broadcast job', error: error.message });
  }
};

// Retry a specific list of recipients from a previous job (admin-selected)
exports.retryBroadcastSpecific = async (req, res) => {
  try {
    const { jobId, recipients = [], delayMs, messageOverride } = req.body;
    const original = broadcastJobs.get(jobId);
    if (!original) {
      return res.status(404).json({ message: 'Original job not found' });
    }

    // Normalize recipients array to list of { to, name }
    const normalizer = (typeof phoneUtils === 'function')
      ? phoneUtils
      : (phoneUtils && typeof phoneUtils.normalizeEgyptPhone === 'function')
        ? phoneUtils.normalizeEgyptPhone
        : null;

    const nameMap = new Map((original.recipients || []).map(r => [r.to, r.name]));

    const selected = [];
    const skipped = [];
    for (const rec of Array.isArray(recipients) ? recipients : []) {
      let to = null;
      try {
        const raw = typeof rec === 'string' ? rec : rec?.to;
        to = normalizer ? normalizer(raw) : String(raw || '').trim();
      } catch (_) {
        to = String((typeof rec === 'string' ? rec : rec?.to) || '').trim();
      }
      if (!to) {
        skipped.push({ to: null, reason: 'empty_recipient' });
        continue;
      }
      if (isPlaceholderPhone(to)) {
        skipped.push({ to, reason: 'placeholder_phone' });
        continue;
      }
      if (!isValidEgyptPhone(to)) {
        skipped.push({ to, reason: 'invalid_phone' });
        continue;
      }
      selected.push({ to, name: nameMap.get(to) || 'مستخدم' });
    }

    if (!selected.length) {
      return res.json({ started: false, message: 'No valid recipients to retry', skipped });
    }

    const effectiveDelay = Number(delayMs ?? original.delayMs ?? ADMIN_BROADCAST_DELAY_MS) || 0;
    const retryJobId = makeJobId();
    const retryJob = {
      id: retryJobId,
      total: selected.length,
      processed: 0,
      failed: 0,
      dryRun: false,
      status: 'running',
      delayMs: effectiveDelay,
      createdAt: new Date(),
      finishedAt: null,
      results: [],
      skipped,
      message: sanitize(messageOverride || original.message),
      recipients: selected,
      retryOf: jobId,
    };
    broadcastJobs.set(retryJobId, retryJob);

    // Process asynchronously
    (async () => {
      try {
        for (let i = 0; i < selected.length; i++) {
          const current = broadcastJobs.get(retryJobId);
          if (!current || current.status !== 'running') break;
          const { to, name } = selected[i];
          const personalized = personalize(retryJob.message, name);
          try {
            await sendWhatsApp(to, personalized);
            retryJob.processed++;
            retryJob.results.push({ to, status: 'sent' });
          } catch (err) {
            retryJob.failed++;
            retryJob.results.push({ to, status: 'failed', code: err.response?.status || null, reason: formatErrorReason(err) });
          }
          if (i < selected.length - 1 && effectiveDelay > 0) {
            await sleep(effectiveDelay);
          }
        }
        const done = broadcastJobs.get(retryJobId);
        if (done && done.status === 'running') {
          done.status = 'completed';
          done.finishedAt = new Date();
        }
      } catch (err) {
        const failedJob = broadcastJobs.get(retryJobId);
        if (failedJob) {
          failedJob.status = 'error';
          failedJob.error = err.message || 'Retry specific broadcast error';
          failedJob.finishedAt = new Date();
        }
      }
    })();

    return res.json({ jobId: retryJobId, total: retryJob.total, started: true, retryOf: jobId, skipped });
  } catch (error) {
    res.status(500).json({ message: 'Error retrying specific recipients', error: error.message });
  }
};

// Direct Premium Management: Assign premium to user without activation request
exports.assignPremium = async (req, res) => {
    try {
        const { userId, planType, months, notes } = req.body;

        if (!userId || !planType || !months) {
            return res.status(400).json({ 
                message: 'userId, planType, and months are required' 
            });
        }

        const validPlans = ['free', 'premium-individual', 'team', 'small-company', 'medium-company', 'large-company'];
        if (!validPlans.includes(planType)) {
            return res.status(400).json({ 
                message: 'Invalid plan type' 
            });
        }

        if (months <= 0 || months > 120) {
            return res.status(400).json({ 
                message: 'Months must be between 1 and 120' 
            });
        }

        const user = await User.findById(userId);
        if (!user) {
            return res.status(404).json({ message: 'User not found' });
        }

        // Calculate expiry date
        const expiryDate = new Date();
        expiryDate.setMonth(expiryDate.getMonth() + months);

        // Update user premium status
        user.premiumActive = true;
        user.premiumExpiry = expiryDate;
        user.planType = planType;

        // Add to premium history
        user.premiumHistory.push({
            planType: planType,
            activatedAt: new Date(),
            expiredAt: expiryDate,
            durationMonths: months,
            assignedBy: req.user._id,
            notes: notes || 'Directly assigned by admin'
        });

        await user.save();

        res.json({ 
            message: 'Premium assigned successfully',
            user: {
                _id: user._id,
                name: user.name,
                email: user.email,
                premiumActive: user.premiumActive,
                premiumExpiry: user.premiumExpiry,
                planType: user.planType
            }
        });
    } catch (error) {
        res.status(500).json({ message: 'Error assigning premium', error: error.message });
    }
};

// Direct Premium Management: Remove premium from user
exports.removePremium = async (req, res) => {
    try {
        const { userId } = req.body;

        if (!userId) {
            return res.status(400).json({ message: 'userId is required' });
        }

        const user = await User.findById(userId);
        if (!user) {
            return res.status(404).json({ message: 'User not found' });
        }

        // Remove premium status
        user.premiumActive = false;
        user.premiumExpiry = null;
        user.planType = 'free';

        await user.save();

        res.json({ 
            message: 'Premium removed successfully',
            user: {
                _id: user._id,
                name: user.name,
                email: user.email,
                premiumActive: user.premiumActive,
                planType: user.planType
            }
        });
    } catch (error) {
        res.status(500).json({ message: 'Error removing premium', error: error.message });
    }
};

// Direct Premium Management: Extend premium for user
exports.extendPremium = async (req, res) => {
    try {
        const { userId, additionalMonths } = req.body;

        if (!userId || !additionalMonths) {
            return res.status(400).json({ 
                message: 'userId and additionalMonths are required' 
            });
        }

        if (additionalMonths <= 0 || additionalMonths > 120) {
            return res.status(400).json({ 
                message: 'Additional months must be between 1 and 120' 
            });
        }

        const user = await User.findById(userId);
        if (!user) {
            return res.status(404).json({ message: 'User not found' });
        }

        // Calculate new expiry date
        const currentExpiry = user.premiumExpiry || new Date();
        const newExpiryDate = new Date(currentExpiry);
        newExpiryDate.setMonth(newExpiryDate.getMonth() + additionalMonths);

        // Update user premium status
        user.premiumActive = true;
        user.premiumExpiry = newExpiryDate;

        // Add to premium history
        user.premiumHistory.push({
            planType: user.planType,
            activatedAt: new Date(),
            expiredAt: newExpiryDate,
            durationMonths: additionalMonths,
            assignedBy: req.user._id,
            notes: 'Extended by admin'
        });

        await user.save();

        res.json({ 
            message: 'Premium extended successfully',
            user: {
                _id: user._id,
                name: user.name,
                email: user.email,
                premiumActive: user.premiumActive,
                premiumExpiry: user.premiumExpiry,
                planType: user.planType
            }
        });
    } catch (error) {
        res.status(500).json({ message: 'Error extending premium', error: error.message });
    }
};